package ru.yandex.reminders.logic.sup;

import lombok.val;
import org.joda.time.Instant;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.reminders.logic.callmeback.in.CallbackRequest;
import ru.yandex.reminders.logic.event.Event;
import ru.yandex.reminders.logic.event.EventId;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.reminder.SendStatus;

public class SupPushManager {
    private static final Logger logger = LoggerFactory.getLogger(SupPushManager.class);

    private final SupPushClient supClient;
    private final XivaPushClient xivaClient;
    private final SupClientSettingsRegistry settingsRegistry;

    public SupPushManager(SupPushClient supClient, XivaPushClient xivaClient, SupClientSettingsRegistry settingsRegistry) {
        this.supClient = supClient;
        this.xivaClient = xivaClient;
        this.settingsRegistry = settingsRegistry;
    }

    public SendStatus pushXiva(EventId id, Reminder reminder) {
        val settings = settingsRegistry.getO(id.getCid());

        if (!settings.isPresent()) {
            return new SendStatus.Failed("Sup is not configured", Option.empty());
        }
        if (!settings.get().getXivaPushToken().isPresent()) {
            return new SendStatus.Failed("Xiva is not configured", Option.empty());
        }
        if (settings.get().getPush().getTtl().exists(ttl -> reminder.getSendDate().plusSeconds(ttl).isBefore(Instant.now()))) {
            return new SendStatus.Failed("expired", Option.empty());
        }
        try {
            return new SendStatus.Sent(xivaClient.push(id.getUid(),
                    XivaPushRequest.of(new XivaPushRequest.Reminder(reminder.getUrl().get())),
                    settings.get().getXivaPushToken().get()), Option.empty());
        } catch (RuntimeException e) {
            logger.error("Failed to push xiva", e);
            return new SendStatus.TryAgain(ExceptionUtils.getAllMessages(e), Option.empty());
        }
    }

    public SendStatus pushSup(CallbackRequest request) {
        val reminder = request.getReminder();
        val settings = settingsRegistry.getO(request.getCid());

        if (!settings.isPresent()) {
            return new SendStatus.Failed("Sup is not configured", Option.empty());
        }
        if (!settings.get().getPush().getTtl().isPresent()) {
            return new SendStatus.Failed("Missing ttl", Option.empty());
        }
        if (!settings.get().getPush().getProject().isPresent()) {
            return new SendStatus.Failed("Missing project", Option.empty());
        }
        if (reminder.getSendDate().plusSeconds(settings.get().getPush().getTtl().get()).isBefore(Instant.now())) {
            return new SendStatus.Failed("expired", Option.empty());
        }
        val push = settings.get().getPush().copy();

        val androidFeatures = push.getAndroidFeatures();
        if (!androidFeatures.flatMap(SupPushRequest.Android::getGroup).isEmpty() && !androidFeatures.flatMap(SupPushRequest.Android::getCounter).isEmpty()) {
            val timestampTtl = 2 * 60 * 60 * 1000;
            push.getAndroidFeatures().get().setTimestamp(Option.of(System.currentTimeMillis() + timestampTtl));
        }

        val receiverExpression = new StringBuffer("tag:uid==" + request.getUid());
        val did = reminder.getSupExtraFields().getDeviceId();
        did.ifPresent(d -> receiverExpression.append(" AND did==").append(d));
        receiverExpression.append(" AND ((app_id IN ('ru.yandex.searchplugin.dev', 'ru.yandex.searchplugin.beta', 'ru.yandex.searchplugin') " +
                        "AND app_version_code >= 7040000) OR (app_id IN ('ru.yandex.mobile.inhouse', 'ru.yandex.mobile') AND app_version_code >= 3051000) " +
                        "OR (app_id IN ('com.yandex.launcher') AND app_version_code >= 2002001))");

        push.setReceiver(Cf.list(receiverExpression.toString()));

        push.setSchedule(Option.of(DateTimeOrNow.dateTime(reminder.getSendDate())));
        push.setAdjustTimeZone(Option.of(false));

        push.setNotification(Option.of(push.getNotification().getOrElse(SupPushRequest.Notification::new)));
        push.setData(Option.of(push.getData().getOrElse(SupPushRequest.PushData::new)));

        val name = request.getData().getName();
        push.getNotification().get().setTitle(
                reminder.getSubject().orElse(push.getNotification().get().getTitle())
                        .orElse(name));

        push.getNotification().get().setBody(
                reminder.getMessage().orElse(push.getNotification().get().getBody())
                        .orElse(request.getData().getDescription()).orElse(name));

        push.getData().get().setPushUri(reminder.getUrl().orElse(push.getData().get().getPushUri()));

        String cid = request.getCid().replace("yandex-", "");
        String pushId = Cf.list("reminders", cid, reminder.getId().toHexString()).mkString(".");

        push.getData().get().setPushId(Option.of(pushId));

        if (!(push.getNotification().get().getTitle().isPresent() && push.getNotification().get().getBody().isPresent())) {
            return new SendStatus.Failed("Missing title or body", Option.empty());
        }
        try {
            return new SendStatus.Sent(supClient.push(push, settings.get().getRequestParameters()), Option.empty());
        } catch (RuntimeException e) {
            logger.error("Failed to push sup", e);
            return new SendStatus.TryAgain(ExceptionUtils.getAllMessages(e), Option.empty());
        }
    }
}
