package ru.yandex.reminders.logic.tv;

import lombok.val;
import org.bson.types.ObjectId;
import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.joda.time.format.ISODateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpUtils;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.reminders.api.reminder.Source;
import ru.yandex.reminders.logic.event.Event;
import ru.yandex.reminders.logic.event.EventData;
import ru.yandex.reminders.logic.event.EventId;
import ru.yandex.reminders.logic.event.EventManager;
import ru.yandex.reminders.logic.event.EventsFilter;
import ru.yandex.reminders.logic.event.SpecialClientIds;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.EventType;
import ru.yandex.reminders.logic.reminder.Reminder;
import ru.yandex.reminders.logic.reminder.ReminderWithEvent;
import ru.yandex.reminders.logic.update.ActionInfo;
import ru.yandex.reminders.logic.update.LockManager;
import ru.yandex.reminders.util.JsonObjectUtils;

public class TvReminderManager {
    @Autowired
    private EventManager eventManager;
    @Autowired
    private LockManager lockManager;

    @Value("${tv.api.host}")
    private String apiHost;
    @Value("${tv.api.timeout.secs}")
    private int apiTimeoutSecs;

    public ListF<TvReminder> getTvProgramsReminders(
            PassportUid uid, ListF<TvProgramId> programIds, ListF<TvChannelId> channelIds, ListF<DateTime> starts)
    {
        ListF<ReminderWithEvent> reminders = findReminders(uid, programIds, channelIds, starts);

        return reminders.map(TvReminderDataConverter::fromReminderWithEvent);
    }

    public void replaceTvProgramReminders(
            PassportUid uid, TvChannelProgramId programId, ListF<TvReminder> reminders, ActionInfo actionInfo)
    {
        lockManager.withLock(uid, EventType.TV_PROGRAM,
                () -> eventManager.createOrUpdateEvent(eventId(uid, programId), eventData(reminders), actionInfo));
    }

    public void addTvProgramReminders(
            PassportUid uid, TvChannelProgramId programId, ListF<TvReminder> reminders, ActionInfo actionInfo)
    {
        lockManager.withLock(uid, EventType.TV_PROGRAM, () -> {
            EventData eventData = eventData(reminders);
            Option<Event> event = eventManager.findEvent(eventId(uid, programId));

            if (event.isDefined()) {
                CollectionF<Reminder> rems = eventData.getReminders().plus(event.get().getReminders())
                        .toMapMappingToKey(r -> r.getEventTs().get()).values();

                eventManager.createOrUpdateEvent(eventId(uid, programId), eventData.withReminders(rems), actionInfo);

            } else {
                eventManager.createOrUpdateEvent(eventId(uid, programId), eventData, actionInfo);
            }
        });
    }

    public void deleteTvProgramsReminders(PassportUid uid,
            ListF<TvProgramId> programIds, ListF<TvChannelId> channelIds, ListF<DateTime> starts, ActionInfo actionInfo)
    {
        lockManager.withLock(uid, EventType.TV_PROGRAM, () -> {
            if (starts.isNotEmpty()) {
                ListF<ObjectId> ids = findReminders(uid, programIds, channelIds, starts)
                        .map(r -> r.getReminder().getId());

                eventManager.deleteReminders(uid, SpecialClientIds.TV, eventsFilter(programIds, channelIds), ids);

            } else {
                eventManager.deleteEvents(uid, SpecialClientIds.TV, eventsFilter(programIds, channelIds));
            }
        });
    }

    public void changeTvProgramsRemindersOffset(PassportUid uid, int offsetMinutes, ActionInfo actionInfo) {
        lockManager.withLock(uid, EventType.TV_PROGRAM, () -> {
            ListF<Event> events = eventManager.findEvents(uid, SpecialClientIds.TV, EventsFilter.any());

            events = events.filter(e -> e.getReminders().exists(r -> !r.getOffset().isSome(offsetMinutes)));

            events.forEach(e -> {
                EventData data = e.getEventData();

                data = data.withReminders(data.getReminders().map(r -> r.getChannel().equals(Channel.SMS)
                        ? TvReminderDataConverter.changeOffset(r, offsetMinutes)
                        : r));

                eventManager.createOrUpdateEvent(e.getId(), data, actionInfo);
            });
        });
    }

    public boolean askTvCanReminderBeSent(Reminder reminder, EventId id) {  // CAL-6982
        val tvReminder = TvReminderDataConverter.fromReminderWithEventId(reminder, id);

        String url = "http://" + apiHost + "/v8/users/reminders/filter";
        url = UrlUtils.addParameters(url, Cf.map(Tuple2List.fromPairs(
                "uid", id.getUid(),
                "channelId", tvReminder.getChannelId().getId(),
                "programId", tvReminder.getProgramId().getId(),
                "startTime", tvReminder.getEventDate().toString(ISODateTimeFormat.dateTimeNoMillis()))));

        val response = ApacheHttpClientUtils.downloadString(url, Timeout.seconds(apiTimeoutSecs));

        try {
            val json = JsonObject.parseObject(response);
            return JsonObjectUtils.getBooleanFieldValueO(json, "canSend").orElseThrow(() -> new RuntimeException("Missing canSend field"));
        } catch (Exception e) {
            throw new RuntimeException("Unexpected server response: " + response, e);
        }
    }

    private static EventsFilter eventsFilter(ListF<TvProgramId> programIds, ListF<TvChannelId> channelIds) {
        EventsFilter filter = EventsFilter.any();

        if (programIds.isNotEmpty()) {
            filter = filter.andByExternalId(programIds.map(TvProgramId.toEventExternalIdF()));
        }
        if (channelIds.isNotEmpty()) {
            filter = filter.andByIdx(channelIds.map(TvChannelId.getIdF()));
        }
        return filter;
    }

    private static EventData eventData(ListF<TvReminder> reminders) {
        return new EventData(
                Option.some(Source.INTERNAL),
                Option.<String>none(), Option.<String>none(), Option.<JsonObject>none(),
                reminders.map(TvReminderDataConverter.toReminderF()));
    }

    private static EventId eventId(PassportUid uid, TvChannelProgramId programId) {
        return new EventId(
                uid, SpecialClientIds.TV,
                programId.getProgramId().toEventExternalId(), programId.getChannelId().getId());
    }

    private ListF<ReminderWithEvent> findReminders(
            PassportUid uid, ListF<TvProgramId> programIds, ListF<TvChannelId> channelIds, ListF<DateTime> starts)
    {
        ListF<Event> events = eventManager.findEvents(uid, SpecialClientIds.TV, eventsFilter(programIds, channelIds));
        ListF<ReminderWithEvent> reminders = events.flatMap(Event.getRemindersWithEventF());

        ListF<Instant> startsTs = starts.map(DateTime::toInstant);
        return starts.isNotEmpty()
                ? reminders.filter(startsTs.containsF().compose(r -> r.getReminder().getEventTs().get()))
                : reminders;
    }
}
