package ru.yandex.reminders.mongodb;

import org.bson.Document;
import org.bson.types.ObjectId;
import org.joda.time.Instant;

import com.mongodb.ReadPreference;
import com.mongodb.WriteConcern;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.storage.mongo.BazingaMongoDbHolder;
import ru.yandex.commune.bazinga.impl.storage.mongo.BazingaMongoUtils;
import ru.yandex.commune.mongo3.MongoCollectionX;
import ru.yandex.commune.mongo3.MongoUtils;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.reminders.worker.FlightShiftSendTask;
import ru.yandex.reminders.worker.ObjectIdActiveUniqueId;
import ru.yandex.reminders.worker.ReminderSendTask;

/**
 * This MDAO was created because there is no appropriate method in BazingaStorage and BazingaTaskManager.
 * <p/>
 * The class directly uses Mongo collection storing Bazinga onetime jobs.
 */
public class BazingaOnetimeJobMdao {

    private final MongoCollectionX<FullJobId, OnetimeJob> onetimeJobs;

    public BazingaOnetimeJobMdao(BazingaMongoDbHolder db) {
        BenderConfiguration configuration = BazingaBender.getConfiguration();

        this.onetimeJobs = new MongoCollectionX<>(
                BazingaMongoUtils.getCollection(db, BazingaMongoUtils.ONETIME_JOBS, 
                    ReadPreference.primaryPreferred(), WriteConcern.ACKNOWLEDGED),
                OnetimeJob.class,
                configuration
        );
    }

    /**
     * @return <code>true</code> if job was rescheduled, <code>false</code> if job was not found
     */
    public boolean rescheduleByActiveUniqueIdAndStatus(String id, JobStatus status, Instant newScheduleTime) {
        return onetimeJobs.findOneAndUpdate(
                new Document(Cf.map("activeUniqueIdentifier", id, "status", MongoUtils.toMongoValue(status))),
                new Document(Cf.map("$set", Cf.map("scheduleTime", MongoUtils.toMongoValue(newScheduleTime)))),
                false, false
        ).isPresent();
    }

    public boolean rescheduleReminderSendTaskByReminderIdAndStatus(
            ObjectId reminderId, JobStatus status, Instant newScheduleTime)
    {
        return rescheduleByActiveUniqueIdAndStatus(reminderIdToActiveUniqueId(reminderId), status, newScheduleTime);
    }

    public static String reminderIdToActiveUniqueId(ObjectId reminderId) {
        return OnetimeJob.generateActiveUniqueIdentifier(ReminderSendTask.TASK_ID,
                new String(BazingaBender.mapper.serializeJson(new ObjectIdActiveUniqueId(reminderId))));
    }

    public static String flightShiftIdToActiveUniqueId(ObjectId flightShiftId) {
        return OnetimeJob.generateActiveUniqueIdentifier(FlightShiftSendTask.TASK_ID,
                new String(BazingaBender.mapper.serializeJson(new ObjectIdActiveUniqueId(flightShiftId))));
    }

    public ListF<OnetimeJob> findByActiveUniqueId(String id) {
        return onetimeJobs.find(new Document("activeUniqueIdentifier", id));
    }

    public ListF<OnetimeJob> findByTaskId(String id) {
        return onetimeJobs.find(new Document("_id.taskId", id));
    }

    public void removeAll() {
        onetimeJobs.deleteAll();
    }

}
