package ru.yandex.reminders.mongodb;

import com.mongodb.event.CommandFailedEvent;
import com.mongodb.event.CommandListener;
import com.mongodb.event.CommandStartedEvent;
import com.mongodb.event.CommandSucceededEvent;
import lombok.val;
import org.apache.commons.lang.StringUtils;
import org.bson.json.JsonMode;
import org.bson.json.JsonWriterSettings;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

public class LoggingCommandListener implements CommandListener {
    public static final int THRESHOLD_MILLIS = 500;

    private static final Logger commandLogger = LoggerFactory.getLogger("ru.yandex.commune.mongo.q");
    private static final JsonWriterSettings JSON_WRITER_SETTINGS = new JsonWriterSettings(JsonMode.STRICT, false);
    private static final int MAX_JSON_LENGTH = 500;

    private final ConcurrentHashMap<Integer, Lazy<String>> commands = new ConcurrentHashMap<>();

    @Override
    public void commandStarted(CommandStartedEvent event) {
        val command = Lazy.withSupplier(() ->
                StringUtils.abbreviate(event.getCommand().toJson(JSON_WRITER_SETTINGS), MAX_JSON_LENGTH));

        commands.put(event.getRequestId(), command);

        val commandName = event.getCommandName();

        if (commandLogger.isDebugEnabled()) {
            commandLogger.debug("{} begin id={} {} {}",
                    commandName,
                    event.getRequestId(),
                    event.getConnectionDescription().getServerAddress(),
                    command.get());
        }
    }

    @Override
    public void commandSucceeded(CommandSucceededEvent event) {
        val commandName = event.getCommandName();

        val command = commands.remove(event.getRequestId());
        val millis = event.getElapsedTime(TimeUnit.MILLISECONDS);
        if (millis > THRESHOLD_MILLIS) {
            if (commandLogger.isEnabledFor(Level.WARN)) {
                // Do not log response values as they may be really huge
                // and converting them to JSON would be a waste of CPU
                commandLogger.warn("{} long id={} {} {}ms {}",
                        commandName,
                        event.getRequestId(),
                        event.getConnectionDescription().getServerAddress(),
                        event.getElapsedTime(TimeUnit.MILLISECONDS),
                        command != null ? command.get() : "command JSON missing");
            }
        } else {
            if (commandLogger.isInfoEnabled()) {
                // Do not log response values as they may be really huge
                // and converting them to JSON would be a waste of CPU
                commandLogger.info("{} done id={} {} {}ms",
                        commandName,
                        event.getRequestId(),
                        event.getConnectionDescription().getServerAddress(),
                        event.getElapsedTime(TimeUnit.MILLISECONDS));
            }
        }
    }

    @Override
    public void commandFailed(CommandFailedEvent event) {
        val command = commands.remove(event.getRequestId());
        if (commandLogger.isEnabledFor(Level.ERROR)) {
            commandLogger.error("{} failed rid{} {} {}ms {}",
                    event.getCommandName(),
                    event.getRequestId(),
                    event.getConnectionDescription().getServerAddress(),
                    event.getElapsedTime(TimeUnit.MILLISECONDS),
                    command != null ? command.get() : "command JSON missing",
                    event.getThrowable());
        }
    }

}
