package ru.yandex.reminders.mongodb;

import com.mongodb.event.CommandFailedEvent;
import com.mongodb.event.CommandListener;
import com.mongodb.event.CommandStartedEvent;
import com.mongodb.event.CommandSucceededEvent;
import io.micrometer.core.instrument.MeterRegistry;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.concurrent.TimeUnit;
import java.util.function.Function;

public class MongoCommandListener implements CommandListener {
    private final static String MONGO_SUCCESS_COMMANDS_METRIC = "reminders.mongo.success.requests";
    private final static String MONGO_FAIL_COMMANDS_METRIC = "reminders.mongo.fail.requests";
    private final static String MONGO_TIMINGS_METRIC = "reminders.mongo.time";

    @Autowired
    private MeterRegistry registry;

    @Override
    public void commandStarted(CommandStartedEvent event) {
    }

    @Override
    public void commandSucceeded(CommandSucceededEvent event) {
        updateMetrics(event::getElapsedTime, MONGO_SUCCESS_COMMANDS_METRIC,
                event.getConnectionDescription().getServerAddress().getHost());
    }

    @Override
    public void commandFailed(CommandFailedEvent event) {
        updateMetrics(event::getElapsedTime, MONGO_FAIL_COMMANDS_METRIC,
                event.getConnectionDescription().getServerAddress().getHost());
    }

    private void updateMetrics(Function<TimeUnit, Long> f, String commandsMetric, String dbhost) {
        registry.counter(commandsMetric + "." + dbhost).increment();
        registry.timer(MONGO_TIMINGS_METRIC).record(f.apply(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
    }
}