package ru.yandex.reminders.mongodb;

import com.mongodb.client.MongoDatabase;
import lombok.val;
import org.bson.types.ObjectId;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.bolts.function.Function0;
import ru.yandex.commune.mongo3.bender.MongoObjectIdMarshaller;
import ru.yandex.commune.mongo3.bender.MongoObjectIdUnmarshaller;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.BenderSettings;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryUtils;
import ru.yandex.misc.bender.config.EmptyMarshallerUnmarshallerFactory;
import ru.yandex.misc.bender.custom.AnyPojoWrapper;
import ru.yandex.misc.bender.custom.AnyPojoWrapperMarshallerUnmarshaller;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.email.bender.EmailMarshaller;
import ru.yandex.misc.email.bender.EmailUnmarshaller;
import ru.yandex.misc.version.AppName;
import ru.yandex.reminders.api.reminder.Source;
import ru.yandex.reminders.logic.event.EventMdao;
import ru.yandex.reminders.logic.flight.Direction;
import ru.yandex.reminders.logic.flight.airport.AirportMdao;
import ru.yandex.reminders.logic.flight.shift.FlightShiftMdao;
import ru.yandex.reminders.logic.flight.shift.FlightShiftSendResultMdao;
import ru.yandex.reminders.logic.reminder.Channel;
import ru.yandex.reminders.logic.reminder.EventType;
import ru.yandex.reminders.logic.reminder.Origin;
import ru.yandex.reminders.logic.reminder.PhoneNumber;
import ru.yandex.reminders.logic.reminder.SendResultMdao;
import ru.yandex.reminders.logic.reminder.SendResultStatus;
import ru.yandex.reminders.logic.user.SettingsMdao;
import ru.yandex.reminders.util.ValueOrObjectMarshallerUnmarshallerFactory;

@Configuration
public class MongoDaoContextConfiguration {

    public static final BenderMapper mapper = new BenderMapper(mongoBenderConfiguration());

    @Autowired
    private MongoDatabase remindersDb;

    @Bean
    public EventMdao eventMdao() {
        return new EventMdao(remindersDb);
    }

    @Bean
    public SendResultMdao sendResultMdao() {
        return new SendResultMdao(remindersDb);
    }

    @Bean
    public SettingsMdao settingsMdao() {
        return new SettingsMdao(remindersDb);
    }

    @Bean
    public FlightShiftMdao flightShiftMdao() {
        return new FlightShiftMdao(remindersDb);
    }

    @Bean
    public FlightShiftSendResultMdao flightShiftSendResultMdao() {
        return new FlightShiftSendResultMdao(remindersDb);
    }

    @Bean
    public AirportMdao airportMdao() {
        return new AirportMdao(remindersDb);
    }

    @Bean
    public PingerMdao pingerMdao(AppName appName) {
        return new PingerMdao(appName, remindersDb, MongoDaoContextConfiguration.mongoBenderConfiguration());
    }

    @Bean
    public MongoCheckWorker mongoCheckWorker(PingerMdao pingerMdao) {
        val mongoCheckWorker = new MongoCheckWorker(pingerMdao, Duration.standardSeconds(30));
        mongoCheckWorker.setDaemon(true);
        mongoCheckWorker.start();
        return mongoCheckWorker;
    }

    public static BenderConfiguration mongoBenderConfiguration() {
        BenderConfiguration basic = new BenderConfiguration(
                new BenderSettings(MembersToBind.WITH_ANNOTATIONS, false),
                new EmptyMarshallerUnmarshallerFactory()
        );
        val mu = new AnyPojoWrapperMarshallerUnmarshaller(
                basic.getSettings(), mapperF());

        basic = CustomMarshallerUnmarshallerFactoryUtils.addMarshallers(
                basic,
                CustomMarshallerUnmarshallerFactoryBuilder.cons()
                        .add(Channel.class, new IntEnumMarshaller(), new IntEnumUnmarshaller<Channel>(Channel.R))
                        .add(EventType.class, new IntEnumMarshaller(), new IntEnumUnmarshaller<EventType>(EventType.R))
                        .add(Direction.class, new IntEnumMarshaller(), new IntEnumUnmarshaller<Direction>(Direction.R))
                        .add(Origin.class, new IntEnumMarshaller(), new IntEnumUnmarshaller<Origin>(Origin.R))
                        .add(Source.class, new IntEnumMarshaller(), new IntEnumUnmarshaller<Source>(Source.R))
                        .add(SendResultStatus.class,
                                new IntEnumMarshaller(),
                                new IntEnumUnmarshaller<>(SendResultStatus.R))
                        .add(ObjectId.class, new MongoObjectIdMarshaller(), new MongoObjectIdUnmarshaller())
                        .add(PhoneNumber.class, new PhoneNumberMarshaller(), new PhoneNumberUnmarshaller())
                        .add(Email.class, new EmailMarshaller(), new EmailUnmarshaller())
                        .add(DateTimeZone.class, new DateTimeZoneMarshaller(), new DateTimeZoneUnmarshaller())
                        .add(CompressibleString.class,
                                new CompressibleObjectMarshaller(CompressibleString.getStringF()),
                                new CompressibleObjectUnmarshaller(CompressibleString.consF()))
                        .add(CompressibleJsonObject.class,
                                new CompressibleObjectMarshaller(CompressibleJsonObject.getStringF()),
                                new CompressibleObjectUnmarshaller(CompressibleJsonObject.fromStringF()))
                        .add(AnyPojoWrapper.class, mu, mu)
        );
        return new BenderConfiguration(basic.getSettings(), CustomMarshallerUnmarshallerFactoryUtils.combine(
                new ValueOrObjectMarshallerUnmarshallerFactory(), basic.getMarshallerUnmarshallerFactory()));
    }

    private static Function0<BenderMapper> mapperF() {
        return () -> mapper;
    }
}
