package ru.yandex.reminders.mongodb;

import com.mongodb.*;
import com.mongodb.client.MongoDatabase;
import lombok.val;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.lang.Check;
import ru.yandex.reminders.api.unistat.UnistatConfiguration;

@Import(
        UnistatConfiguration.class
)
@Configuration
public class MongoDbContextConfiguration {
    private static final int DEFAULT_PORT = 27017;

    @Value("${mongo.hosts}")
    private String hosts;
    @Value("${mongo.connections.per.host}")
    private int connectionsPerHost;
    @Value("${mongo.wait.multiplier}")
    private int multiplier;
    @Value("${mongo.socket.timeoutMs}")
    private int socketTimeoutMs;
    @Value("${mongo.connect.timeoutMs}")
    private int connectTimeoutMs;
    @Value("${mongo.max.wait.timeMs}")
    private int maxWaitTimeMs;
    @Value("${mongo.write.concern}")
    private String writeConcernStr;
    @Value("${mongo.read.preference}")
    private String readPreferenceStr;

    @Value("${mongo.auth.username}")
    private String username;
    @Value("${mongo.auth.password}")
    private String password;

    @Value("${mongo.db.reminders}")
    private String remindersDb;
    @Value("${mongo.db.bazinga}")
    private String bazingaDb;

    @Bean
    public MongoClient mongoClient() {
        val seeds = Cf.String.splitF(",").apply(hosts).map(strToAddressF());

        if (username.isEmpty()) {
            return new MongoClient(seeds, mongoClientOptions());
        }
        ListF<MongoCredential> credentials = Cf.list(remindersDb, bazingaDb, "admin")
                .map(db -> MongoCredential.createMongoCRCredential(username, db, password.toCharArray()));

        return new MongoClient(seeds, credentials, mongoClientOptions());
    }

    @Bean
    public MongoDatabase remindersDb(MongoClient mongoClient) {
        return mongoClient.getDatabase(remindersDb);
    }

    @Bean
    public MongoDatabase bazingaDb(MongoClient mongoClient) {
        return mongoClient.getDatabase(bazingaDb);
    }

    @Bean
    public MongoCommandListener mongoCommandListener() {
        return new MongoCommandListener();
    }

    protected MongoClientOptions mongoClientOptions() {
        val writeConcern = WriteConcern.valueOf(writeConcernStr);
        Check.notNull(writeConcern, "Unknown writeConcert=" + writeConcernStr + " specified.");

        return MongoClientOptions.builder()
                .connectionsPerHost(connectionsPerHost)
                .threadsAllowedToBlockForConnectionMultiplier(multiplier)
                .socketTimeout(socketTimeoutMs)
                .connectTimeout(connectTimeoutMs)
                .maxWaitTime(maxWaitTimeMs)
                .writeConcern(writeConcern)
                .readPreference(ReadPreference.valueOf(readPreferenceStr))
                .addCommandListener(mongoCommandListener())
                .addCommandListener(new LoggingCommandListener())
                .build();
    }

    public static Function<String, ServerAddress> strToAddressF() {
        return s -> {
            int pos = s.indexOf(":");
            return pos >= 0
                    ? new ServerAddress(s.substring(0, pos), Integer.parseInt(s.substring(pos + 1)))
                    : new ServerAddress(s, DEFAULT_PORT);
        };
    }

}
