package ru.yandex.reminders.util;

import lombok.val;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.ip.Host;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.lang.Validate;

import java.util.regex.Pattern;

public class HostnameUtils extends ru.yandex.misc.net.HostnameUtils {
    private static final Pattern HOST_ID_PATTERN =
            Pattern.compile("^[a-z-_]+(\\d+[a-z]?)(?:\\.|$)", Pattern.CASE_INSENSITIVE);

    // http://data.iana.org/TLD/tlds-alpha-by-domain.txt version 2014063000 IDNA excluded
    private static final String ROOT_DOMAIN_NAMES_PATTERN_STR = "ac|academy|accountants|active|actor|ad|ae|aero|af|ag" +
            "|agency|ai|airforce|al|am|an|ao|aq|ar|archi|army|arpa|as|asia|associates|at|attorney|au|audio|autos" +
            "|aw|ax|axa|az|ba|bar|bargains|bayern|bb|bd|be|beer|berlin|best|bf|bg|bh|bi|bid|bike|bio|biz|bj|black" +
            "|blackfriday|blue|bm|bmw|bn|bo|boutique|br|brussels|bs|bt|build|builders|buzz|bv|bw|by|bz|bzh|ca|cab" +
            "|camera|camp|capetown|capital|cards|care|career|careers|cash|cat|catering|cc|cd|center|ceo|cf|cg|ch" +
            "|cheap|christmas|church|ci|citic|ck|cl|claims|cleaning|clinic|clothing|club|cm|cn|co|codes|coffee" +
            "|college|cologne|com|community|company|computer|condos|construction|consulting|contractors|cooking" +
            "|cool|coop|country|cr|credit|creditcard|cruises|cu|cv|cw|cx|cy|cz|dance|dating|de|degree|democrat" +
            "|dental|dentist|desi|diamonds|digital|directory|discount|dj|dk|dm|dnp|do|domains|durban|dz|ec|edu" +
            "|education|ee|eg|email|engineer|engineering|enterprises|equipment|er|es|estate|et|eu|eus|events" +
            "|exchange|expert|exposed|fail|farm|feedback|fi|finance|financial|fish|fishing|fitness|fj|fk|flights" +
            "|florist|fm|fo|foo|foundation|fr|frogans|fund|furniture|futbol|ga|gal|gallery|gb|gd|ge|gf|gg|gh|gi" +
            "|gift|gives|gl|glass|global|globo|gm|gmo|gn|gop|gov|gp|gq|gr|graphics|gratis|green|gripe|gs|gt|gu" +
            "|guide|guitars|guru|gw|gy|hamburg|haus|hiphop|hiv|hk|hm|hn|holdings|holiday|homes|horse|host|house" +
            "|hr|ht|hu|id|ie|il|im|immobilien|in|industries|info|ink|institute|insure|int|international|investments" +
            "|io|iq|ir|is|it|je|jetzt|jm|jo|jobs|joburg|jp|juegos|kaufen|ke|kg|kh|ki|kim|kitchen|kiwi|km|kn|koeln" +
            "|kp|kr|kred|kw|ky|kz|la|land|lawyer|lb|lc|lease|li|life|lighting|limited|limo|link|lk|loans|london" +
            "|lotto|lr|ls|lt|lu|luxe|luxury|lv|ly|ma|maison|management|mango|market|marketing|mc|md|me|media|meet" +
            "|menu|mg|mh|miami|mil|mini|mk|ml|mm|mn|mo|mobi|moda|moe|monash|mortgage|moscow|motorcycles|mp|mq|mr" +
            "|ms|mt|mu|museum|mv|mw|mx|my|mz|na|nagoya|name|navy|nc|ne|net|neustar|nf|ng|nhk|ni|ninja|nl|no|np|nr" +
            "|nu|nyc|nz|okinawa|om|onl|org|organic|ovh|pa|paris|partners|parts|pe|pf|pg|ph|photo|photography|photos" +
            "|physio|pics|pictures|pink|pk|pl|plumbing|pm|pn|post|pr|press|pro|productions|properties|ps|pt|pub|pw" +
            "|py|qa|qpon|quebec|re|recipes|red|rehab|reise|reisen|ren|rentals|repair|report|republican|rest|reviews" +
            "|rich|rio|ro|rocks|rodeo|rs|ru|ruhr|rw|ryukyu|sa|saarland|sb|sc|schule|scot|sd|se|services|sexy|sg|sh" +
            "|shiksha|shoes|si|singles|sj|sk|sl|sm|sn|so|social|software|sohu|solar|solutions|soy|space|sr|st|su" +
            "|supplies|supply|support|surf|surgery|sv|sx|sy|systems|sz|tattoo|tax|tc|td|technology|tel|tf|tg|th" +
            "|tienda|tips|tirol|tj|tk|tl|tm|tn|to|today|tokyo|tools|town|toys|tp|tr|trade|training|travel|tt|tv|tw" +
            "|tz|ua|ug|uk|university|uno|us|uy|uz|va|vacations|vc|ve|vegas|ventures|versicherung|vet|vg|vi|viajes" +
            "|villas|vision|vlaanderen|vn|vodka|vote|voting|voto|voyage|vu|wang|watch|webcam|website|wed|wf|wien" +
            "|wiki|works|ws|wtc|wtf|xxx|xyz|yachts|ye|yokohama|yt|za|zm|zone|zw";

    private static final Pattern HOSTNAME_PATTERN = Pattern.compile(
            "^(?:(?:[a-z]|[a-z][a-z0-9-]*[a-z0-9])\\.)+(?:" + ROOT_DOMAIN_NAMES_PATTERN_STR + ")$",
            Pattern.CASE_INSENSITIVE);

    public static Option<String> getLocalhostId() {
        return getHostId(ru.yandex.misc.net.HostnameUtils.localHostname());
    }

    public static Option<String> getHostId(String hostname) {
        val matcher = HOST_ID_PATTERN.matcher(hostname);
        return matcher.find() ? Option.of(matcher.group(1)) : Option.empty();
    }

    public static boolean looksLikeHostname(String str) {
        return HOSTNAME_PATTERN.matcher(str).matches();
    }

    public static void validateHostOrHostPort(String host) {
        try {
            Host.parse(host);
        } catch (IllegalArgumentException e) {
            try {
                HostPort.parse(host);
            } catch (IllegalArgumentException e2) {
                Validate.fail("'" + host + "' can't be parsed as host nor host and port");
            }
        }
    }
}
