package ru.yandex.reminders.util.jsonld;

import lombok.val;
import org.joda.time.Duration;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.format.ISOPeriodFormat;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.Iso8601;
import ru.yandex.reminders.util.JsonObjectUtils;

import java.util.Optional;

public class SchemaOrgEventDataExtractor {
    private static final Logger logger = LoggerFactory.getLogger(SchemaOrgEventDataExtractor.class);

    private static final SetF<String> KNOWN_EVENT_TYPES = Cf.set( // XXX: obtain and parse owl
            "Event", "BusinessEvent", "ChildrensEvent", "ComedyEvent",
            "DanceEvent", "DeliveryEvent", "EducationEvent",
            "Festival", "FoodEvent", "LiteraryEvent", "MusicEvent", "PublicationEvent",
            "SaleEvent", "SocialEvent", "SportsEvent", "TheaterEvent", "UserInteraction", "VisualArtsEvent");

    public static ExtractedEventData extract(JsonObject object) {
        try {
            return extractUnsafe(object);
        } catch (Exception e) {
            logger.error("failed to extract data from json", e);
            return ExtractedEventData.empty();
        }
    }

    static ExtractedEventData extractUnsafe(JsonObject object) {
        object = SchemaOrgJsonLdProcessor.compact(object);

        if (!stringValue(object, "@type").filter(KNOWN_EVENT_TYPES.containsF()).isPresent()) {
            return ExtractedEventData.empty();
        }
        return new ExtractedEventData(
                stringValue(object, "name"),
                stringValue(object, "description"),
                stringValue(object, "url"),
                stringValue(object, "image"),
                stringValue(object, "startDate").flatMap(parseDateSafeF()),
                stringValue(object, "endDate").flatMap(parseDateSafeF()),
                stringValue(object, "duration").flatMap(parseDurationSafeF()));
    }

    private static Optional<String> stringValue(JsonObject object, String fieldName) {
        val valueO = object.getO(fieldName);
        if (!valueO.isPresent()) {
            return Optional.empty();
        }
        JsonValue value = valueO.get();
        if (value instanceof JsonString) {
            return JsonObjectUtils.getStringFieldValueO(object, fieldName);
        }
        if (value instanceof JsonObject) {
            return stringValue((JsonObject) value, "@value");
        }
        // XXX can be specified more than one value, e.g. multiple startDates or multiple @language
        return Optional.empty();
    }

    private static Optional<Either<LocalDate, LocalDateTime>> parseDateSafe(String value) {
        try {
            return Optional.of(Either.right(Iso8601.parseLocalDateTime(value)));
        } catch (IllegalArgumentException ignored) {}
        try {
            return Optional.of(Either.left(Iso8601.parseLocalDate(value)));
        } catch (IllegalArgumentException ignored) {}

        logger.warn("failed to parse {} as date", value);
        return Optional.empty();
    }

    private static Optional<Duration> parseDurationSafe(String value) {
        try {
            return Optional.of(ISOPeriodFormat.standard().parsePeriod(value).toStandardDuration());
        } catch (IllegalArgumentException | UnsupportedOperationException ignored) {}

        logger.warn("failed to parse {} as duration", value);
        return Optional.empty();
    }

    private static Function<String, Optional<Either<LocalDate, LocalDateTime>>> parseDateSafeF() {
        return SchemaOrgEventDataExtractor::parseDateSafe;
    }

    private static Function<String, Optional<Duration>> parseDurationSafeF() {
        return SchemaOrgEventDataExtractor::parseDurationSafe;
    }
}
