package ru.yandex.reminders.util.task;

import lombok.val;
import org.springframework.context.ApplicationContext;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.log.reqid.RequestIdStack;

public class ManualTaskLauncher {
    private final MapF<String, ManualTask> taskByBeanName;
    private TaskThread taskThread;

    public ManualTaskLauncher(ApplicationContext applicationContext) {
        this.taskByBeanName = findTasks(applicationContext);
    }

    public synchronized String runTask(String taskBeanName) {
        if (taskThread == null) {
            Option<ManualTask> task = taskByBeanName.getO(taskBeanName);
            if (task.isEmpty()) {
                return String.format("Task bean with name '%s' not found in applicationContext.", taskBeanName);
            }
            taskThread = createThread(taskBeanName, task.get());
            taskThread.start();
            return String.format("Thread %s started for task %s.", taskThread.getName(), taskBeanName);
        } else {
            return String.format(
                    "Only one task can run. Task %s is now running. Wait until it finishes and retry.\n\n%s",
                    taskThread.getTaskBeanName(),
                    taskStatusesToString());
        }
    }

    public synchronized String status() {
        String result;
        if (taskThread != null && taskThread.isAlive()) {
            result = String.format("Task %s is in progress.", taskThread.getTaskBeanName());
        } else {
            result = "No running task.";
        }
        return result + "\n\n" + taskStatusesToString();
    }

    public synchronized String stopRunningTask() {
        return taskThread != null && taskThread.isAlive() ? taskThread.stopTask() : "No running task to stop.";
    }

    public synchronized String terminateRunningTask() {
        return taskThread != null && taskThread.isAlive() ? taskThread.killTask() : "No running task to stop.";
    }

    private MapF<String, ManualTask> findTasks(ApplicationContext applicationContext) {
         return Cf.wrap(applicationContext.getBeansOfType(ManualTask.class));
    }

    private TaskThread createThread(String taskBeanName, final ManualTask manualTask) {
        return new TaskThread(taskBeanName, manualTask, () -> doRunTask(manualTask));
    }

    private void doRunTask(ManualTask manualTask) {
        val requestIdHandle = RequestIdStack.push();
        try {
            manualTask.execute();
        } finally {
            requestIdHandle.popSafely();
            taskThread = null;
        }
    }

    private String taskStatusesToString() {
        return taskByBeanName.mapValues(getTaskStatusF().andThen(ObjectUtils.toStringM())).entries().sortedBy1()
                .map((Function<Tuple2<String, String>, Object>) t -> String.format("task: %s, status: %s", t.get1(), t.get2())).mkString("\n");
    }

    private static Function<ManualTask, TaskStatus> getTaskStatusF() {
        return ManualTask::getStatus;
    }

}
