package ru.yandex.reminders.util.task;

import org.joda.time.Duration;
import org.joda.time.Instant;
import ru.yandex.bolts.collection.Option;

public class TaskStatus {

    private final Instant started;
    private final Option<Instant> finished;
    private final Duration elapsed;
    private final State state;
    private final String msg;

    protected TaskStatus(Instant started, Option<Instant> finished, Duration elapsed, TaskStatus.State state,
                         String msg) {
        this.started = started;
        this.finished = finished;
        this.elapsed = elapsed;
        this.state = state;
        this.msg = msg;
    }

    public static TaskStatus running(Instant started, String msg) {
        return new TaskStatus(
                started, Option.empty(), new Duration(started, Instant.now()), State.RUNNING, msg);
    }

    public static TaskStatus success(Instant started, String msg) {
        Instant now = Instant.now();
        return new TaskStatus(
                started, Option.of(now), new Duration(started, now), State.SUCCESS, msg);
    }

    public static TaskStatus fail(Instant started, String msg) {
        Instant now = Instant.now();
        return new TaskStatus(
                started, Option.of(now), new Duration(started, now), State.FAIL, msg);
    }

    public static TaskStatus stopped(Instant started, String msg) {
        Instant now = Instant.now();
        return new TaskStatus(
                started, Option.of(now), new Duration(started, now), State.STOPPED, msg);
    }

    public boolean isFinished() {
        return state != State.RUNNING;
    }

    public String getMsg() {
        return msg;
    }

    @Override
    public String toString() {
        return String.format(
                "%s: %s (elapsed=%s, started at=%s%s)",
                state, msg, elapsed, started,
                finished.isEmpty() ? "" : String.format(
                        ", %s at=%s", state == State.STOPPED ? "stopped" : "finished", finished.get())
        );
    }

    public static enum State {
        RUNNING,
        SUCCESS,
        FAIL,
        STOPPED,
        ;
    }
}
