package ru.yandex.reminders.worker;

import lombok.val;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.commune.alive2.AliveAppsHolder;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

public class CheckVersionServlet extends HttpServlet {
    private final AliveAppsHolder holder;
    private final String projectVersion;
    private volatile STATE state = STATE.START;
    private static final String PATTERN = "\\d+";
    private static final Logger logger = LoggerFactory.getLogger(CheckVersionServlet.class);

    private enum STATE {
        START,
        ALIVE,
        FAIL
    }

    public CheckVersionServlet(AliveAppsHolder holder, String projectVersion) {
        this.holder = holder;
        this.projectVersion = projectVersion;
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        switch (state) {
            case FAIL:
                resp.setStatus(HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
                return;
            case START:
                val workerApps = holder.aliveAppsO()
                        .getOrElse(ArrayListF::new)
                        .filter(app -> "worker".equals(app.getAppName()))
                        .filter(app -> isOtherVersionNewer(projectVersion, app.getVersion()));
                if (!workerApps.isEmpty()) {
                    state = STATE.FAIL;
                    logger.error("Found other workers with newer version {}, this worker version is {}",
                            workerApps.mkString("\n"), projectVersion);
                    resp.setStatus(HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
                    return;
                } else holder.aliveAppsO().ifPresent(list -> {
                    logger.info("Other workers have the same or older version, therefore starting functioning normally...");
                    state = STATE.ALIVE;
                });
            default:
                resp.getWriter().write("/ok ping");
        }
    }

    private static boolean isOtherVersionNewer(String projectVersion, String other) {
        // Если обе версии являются числами, то сравниваем их как числа,
        // в противноом случае сравниваем лексикографически.
        // При этом, если версия нового воркера является числом, то версию нечисловую других воркеров мы игнорируем,
        // для переходного периода, когда есть одновременно QLoud и старые машинки
        if (projectVersion.matches(PATTERN)) {
            if (other.matches(PATTERN)) {
                return Integer.valueOf(other) > Integer.valueOf(projectVersion);
            }
            return false;
        }
        return projectVersion.compareTo(other) < 0;
    }
}