package ru.yandex.reminders.worker;

import lombok.val;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleLinear;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.reminders.logic.flight.shift.FlightShiftSendManager;

public class FlightShiftSendTask extends OnetimeTask {
    public static final TaskId TASK_ID = new TaskId("send-flight-shift");

    private final ReschedulePolicy reschedulePolicy;

    @Autowired
    private FlightShiftSendManager flightShiftSendManager;
    @Autowired
    private EnvironmentType environmentType;

    public FlightShiftSendTask() {
        super(FlightShiftSendTaskParameters.class);
        this.reschedulePolicy = new CompoundReschedulePolicy(
                new RescheduleLinear(Duration.standardSeconds(20), 5),
                new RescheduleConstant(Duration.standardMinutes(2), Integer.MAX_VALUE)
        );
    }

    public FlightShiftSendTask(FlightShiftSendTaskParameters parameters) {
        this();
        setParameters(parameters);
    }

    @Override
    public TaskId id() {
        return TASK_ID;
    }

    @Override
    public int priority() {
        return 1; // higher priority than normal reminders
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(10);
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return reschedulePolicy;
    }

    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return FlightShiftSendActiveUniqueIdConverter.class;
    }

    @Override
    public void execute(ExecutionContext context) {
        if (environmentType == EnvironmentType.TESTS) return;

        val parameters = (FlightShiftSendTaskParameters) getParameters();

        val status = flightShiftSendManager.sendFlightShift(parameters.getFlightShiftId());
        context.setExecutionInfo(status);
        if (status.isTryAgain()) {
            throw new BazingaTryAgainLaterException("at least 1 sms was sent with tryAgain status or failed with exception");
        }
    }
}
