package ru.yandex.reminders.worker;

import lombok.val;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleLinear;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.reminders.logic.reminder.ReminderSendManager;

public class ReminderSendTask extends OnetimeTask {
    public static final TaskId TASK_ID = new TaskId("send-reminder");

    private final ReschedulePolicy reschedulePolicy;

    @Autowired
    private ReminderSendManager reminderSendManager;
    @Autowired
    private EnvironmentType environmentType;

    public ReminderSendTask() {
        super(ReminderSendTaskParameters.class);
        this.reschedulePolicy = new CompoundReschedulePolicy(
                new RescheduleLinear(Duration.standardSeconds(20), 3),
                new RescheduleLinear(Duration.standardMinutes(2), 15),
                new RescheduleConstant(Duration.standardMinutes(30), Integer.MAX_VALUE)
        );
    }

    public ReminderSendTask(ReminderSendTaskParameters parameters) {
        this();
        setParameters(parameters);
    }

    @Override
    public TaskId id() {
        return TASK_ID;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(5);
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return reschedulePolicy;
    }

    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return ReminderSendActiveUniqueIdConverter.class;
    }

    @Override
    public void execute(ExecutionContext context) {
        if (environmentType == EnvironmentType.TESTS) return;

        ReminderSendTaskParameters parameters = (ReminderSendTaskParameters) getParameters();

        val eventId = parameters.getEventId();
        val reminderId = parameters.getReminderId();

        val status = reminderSendManager.sendReminder(eventId, reminderId);
        if (status.isSent()) {
            context.setExecutionInfo(ReminderSentInfo.sent(status.asSent().getMessageId().toOptional()));
        } else if (status.isFailed()) {
            context.setExecutionInfo(ReminderSentInfo.failed(status.asFailed().getFailureReason()));
        } else if (status.isTryAgain()) {
            throw new BazingaTryAgainLaterException(status.asTryAgain().getMessage());
        } else {
            throw new RuntimeException("Unknown status class=" + status.getClass().getName());
        }
    }

}
