package ru.yandex.reminders.worker;

import lombok.val;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.impl.SingletonMap;
import ru.yandex.commune.actor.typed.dynamic.DynamicConfigurationTypedServer;
import ru.yandex.commune.alive2.AliveAppsContextConfiguration;
import ru.yandex.commune.alive2.AliveAppsHolder;
import ru.yandex.commune.bazinga.BazingaActorServerHolder;
import ru.yandex.commune.bazinga.BazingaConfiguration;
import ru.yandex.commune.bazinga.admin.BazingaAdminAddressResolver;
import ru.yandex.commune.bazinga.admin.BazingaAdminAddressResolverStandard;
import ru.yandex.commune.bazinga.context.BazingaContextConfiguration;
import ru.yandex.commune.bazinga.impl.controller.BazingaControllerConfiguration;
import ru.yandex.commune.bazinga.impl.storage.mongo.BazingaMongoStorageCleaner;
import ru.yandex.commune.bazinga.impl.storage.mongo.MongoBazingaStorage;
import ru.yandex.commune.bazinga.impl.worker.BazingaWorkerConfiguration;
import ru.yandex.commune.bazinga.impl.worker.OnetimeTaskLogConfiguration;
import ru.yandex.commune.bazinga.impl.worker.WorkerLogConfiguration;
import ru.yandex.commune.bazinga.scheduler.TaskCategory;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.ip.InternetDomainName;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.reminders.boot.RemindersZkContextConfiguration;
import ru.yandex.reminders.mongodb.BazingaMongoClientContextConfiguration;
import ru.yandex.reminders.mongodb.MongoDbContextConfiguration;
import ru.yandex.scheduler.QuorumWorkerChooser;

import javax.annotation.Resource;

@Configuration
@Import({
        BazingaContextConfiguration.class,
        BazingaMongoClientContextConfiguration.class,
        MongoDbContextConfiguration.class,
        RemindersZkContextConfiguration.class,
        AliveAppsContextConfiguration.class
})
public class RemindersBazingaContextConfiguration {
    private static final String QUORUM_KEY = "QUORUM";

    @Qualifier("zkRoot")
    @Autowired
    private ZkPath zkRoot;
    @Autowired
    private EnvironmentType environmentType;
    @Resource
    private MongoBazingaStorage bazingaStorage;

    @Value("${bazinga.port}")
    private int bazingaPort;
    @Value("${bazinga.log.dir}")
    private File2 bazingaLogDir;

    private ZkPath bazingaZkPath() {
        return zkRoot.child("bazinga");
    }

    @Bean
    public BazingaConfiguration bazingaConfiguration() {
        return new BazingaConfiguration(
                bazingaZkPath(),
                new InternetDomainName(HostnameUtils.localHostname()),
                IpPort.cons(bazingaPort)
        );
    }

    @Bean
    public BazingaControllerConfiguration bazingaControllerConfiguration(AliveAppsHolder holder,
                                                                         @Value("${worker.quorum.poll.delay}") int pollDelay,
                                                                         @Value("${worker.quorum.gc.delay}") int gcDelay) {
        val chooser = new QuorumWorkerChooser(holder);
        val map = new SingletonMap(QUORUM_KEY, chooser) {
            @Override
            public Option getO(Object key) {
                return Option.of(chooser);
            }
        };

        return new BazingaControllerConfiguration(Duration.standardSeconds(pollDelay), Duration.standardSeconds(gcDelay), map);
    }

    @Bean
    public BazingaWorkerConfiguration bazingaWorkerConfiguration() {
        val limit = environmentType == EnvironmentType.PRODUCTION
                ? DataSize.fromGigaBytes(3)
                : DataSize.fromMegaBytes(100);

        return new BazingaWorkerConfiguration(
                Cf.list(TaskCategory.DEFAULT),
                new WorkerLogConfiguration(
                        bazingaLogDir,
                        Option.of(limit),
                        Option.empty(),
                        Option.of(limit),
                        Option.of(100),
                        Option.of(OnetimeTaskLogConfiguration.DEFAULT)
                ),
                Option.empty(),
                64,
                0,
                64 * 6,
                0
        );
    }

    @Bean
    public BazingaMongoStorageCleaner bazingaMongoStorageCleaner() {
        return new BazingaMongoStorageCleaner(bazingaStorage);
    }

    @Bean
    public BazingaActorServerHolder bazingaActorServerHolder() {
        return new BazingaActorServerHolder(typedServer());
    }

    @Bean
    public DynamicConfigurationTypedServer typedServer() {
        return new DynamicConfigurationTypedServer(IpPort.cons(bazingaPort));
    }

    @Bean
    public BazingaAdminAddressResolver bazingaAdminAddressResolver() {
        return new BazingaAdminAddressResolverStandard("reminders-worker");
    }
}
