package ru.yandex.reminders.worker;

import lombok.val;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import ru.yandex.commune.a3.action.result.ApplicationInfo;
import ru.yandex.commune.a3.action.result.ApplicationResultType;
import ru.yandex.commune.a3.action.result.type.MediaType;
import ru.yandex.commune.alive2.AliveAppsHolder;
import ru.yandex.misc.version.AppName;
import ru.yandex.misc.version.Version;
import ru.yandex.misc.web.servlet.PingConfigurableServlet;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;
import ru.yandex.reminders.api.unistat.ApiUnistatContextConfiguration;
import ru.yandex.reminders.boot.RemindersAppContextConfiguration;
import ru.yandex.reminders.logic.flight.airport.AirportContextConfiguration;
import ru.yandex.reminders.logic.flight.airport.AirportManager;
import ru.yandex.reminders.tvm.TvmClientConfiguration;
import ru.yandex.reminders.util.task.ManualTaskContextConfiguration;

@Configuration
@Import({TvmClientConfiguration.class,
    RemindersAppContextConfiguration.class,
    RemindersBazingaContextConfiguration.class,
    ManualTaskContextConfiguration.class,
    AirportContextConfiguration.class,
    ApiUnistatContextConfiguration.class,
})
public class WorkerAppContextConfiguration {

    @Bean
    public SingleWarJetty apiJetty(
            @Value("${api.jetty.port}") int port,
            @Value("${api.jetty.maxThreads}") int maxThreads,
            AliveAppsHolder holder, Version version)
    {
        val jetty = new SingleWarJetty();
        jetty.setHttpPort(port);
        jetty.setMaxThreads(maxThreads);
        jetty.setLookupServletsInContext(false);

        jetty.addServletMapping("/ping/*", apiPingConfigurableServlet());
        jetty.addServletMapping("/check-revision", new CheckVersionServlet(holder, version.getProjectVersion()));

        return jetty;
    }

    @Bean
    public PingConfigurableServlet apiPingConfigurableServlet() {
        return new PingConfigurableServlet();
    }

    @Bean
    public ApplicationInfo applicationInfo(AppName appName, Version version) {
        return new ApplicationInfo(appName.appName(), version.getProjectVersion());
    }

    @Bean
    public ApplicationResultType applicationResultType() {
        return new ApplicationResultType(MediaType.APPLICATION_JSON);
    }

    @Bean
    public ReminderSendTask reminderSendTask() {
        return new ReminderSendTask();
    }

    @Bean
    public FlightShiftSendTask flightShiftSendTask() {
        return new FlightShiftSendTask();
    }

    @Bean
    public DeleteCloudApiFlightsTask deleteCloudApiFlightsTask() {
        return new DeleteCloudApiFlightsTask();
    }

    @Bean
    public AirportsImportTask airportsImportTask(
            AirportManager airportManager,
            @Value("${rasp.timeout.secs}") int timeoutSecs)
    {
        return new AirportsImportTask(airportManager, Duration.standardSeconds(timeoutSecs * 3));
    }
}
