#pragma once

#include "types.h"
#include <json/value.h>
#include <json/writer.h>

namespace rcache {

struct cache_entry
{
    cache_entry() = default;

    cache_entry(const string& value, const duration& ttl)
        : value(value), expires_at(clock::now() + ttl)
    {
    }

    auto ttl() const
    {
        return time_traits::get_seconds_count(expires_at - clock::now());
    }

    string to_string() const
    {
        Json::Value json;
        json["value"] = value;
        json["ttl"] = static_cast<int64_t>(ttl());
        Json::FastWriter writer;
        return writer.write(json);
    }

    string value;
    time_point expires_at;
};

class local_cache
{
public:
    local_cache(const duration& cleanup_interval)
        : cleanup_interval_(cleanup_interval), last_cleanup_ts_(clock::now())
    {
    }

    optional<cache_entry> get(const string& key)
    {
        clear_if_needed();
        auto it = cache_.find(key);
        if (it == cache_.end() || clock::now() >= it->second.expires_at) return {};
        return it->second;
    }

    void set(const string& key, const cache_entry& entry)
    {
        clear_if_needed();
        cache_[key] = entry;
    }

    std::size_t size()
    {
        clear_if_needed();
        return cache_.size();
    }

private:
    void clear_if_needed()
    {
        auto now = clock::now();
        if (now < last_cleanup_ts_ + cleanup_interval_)
        {
            return;
        }
        last_cleanup_ts_ = now;
        for (auto it = cache_.begin(); it != cache_.end();)
        {
            if (now >= it->second.expires_at)
            {
                it = cache_.erase(it);
            }
            else
            {
                ++it;
            }
        }
    }

    std::map<string, cache_entry> cache_;
    duration cleanup_interval_;
    time_point last_cleanup_ts_;
};

}