#include "local_cache.h"
#include <ymod_webserver/server.h>
#include <ymod_httpclient/cluster_client.h>
#include <yplatform/module_registration.h>
#include <yplatform/find.h>
#include <map>

#include <yplatform/yield.h>

namespace rcache {

bool is_localhost(const string& hostname)
{
    if (hostname == "localhost") return true;
    using resolver_type = boost::asio::ip::tcp::resolver;
    boost::asio::io_service io;
    resolver_type resolver(io);
    resolver_type::query hostname_query(hostname, "");
    for (auto fwd_it = resolver.resolve(hostname_query); fwd_it != resolver_type::iterator();
         ++fwd_it)
    {
        for (auto rev_it = resolver.resolve(*fwd_it); rev_it != resolver_type::iterator(); ++rev_it)
        {
            if (rev_it->host_name() == boost::asio::ip::host_name()) return true;
        }
    }
    return false;
}

class module : public yplatform::module
{
public:
    module(yplatform::reactor& reactor, const yplatform::ptree& conf)
        : io_(reactor.io()), cache_(conf.get<duration>("cleanup_interval"))
    {
        for (size_t i = 0; i < reactor.size(); ++i)
        {
            if (reactor[i]->size() != 1)
            {
                throw std::runtime_error("rcache is optimized for single-thread reactors - set "
                                         "pool_count=N and io_threads=1");
            }
        }

        int my_port = conf.get<int>("port");
        yhttp::cluster_client::settings http_client_settings;
        if (auto http_conf = conf.get_child_optional("http"))
        {
            http_client_settings.parse_ptree(*http_conf);
        }
        http_client_settings.nodes = { "" };
        auto nodes_conf = conf.equal_range("nodes");
        for (auto& [name, node] : boost::make_iterator_range(nodes_conf))
        {
            auto host = node.get<string>("host");
            auto port = node.get("port", 80);
            auto ssl = node.get<bool>("ssl", false);
            if (port == my_port && is_localhost(host)) continue;
            http_client_settings.nodes[0] =
                (ssl ? "https://" : "http://") + host + ":" + std::to_string(port);
            YLOG_L(info) << "add node " << http_client_settings.nodes[0];
            nodes_.emplace_back(
                std::make_shared<yhttp::cluster_client>(reactor, http_client_settings));
        }
    }

    void init()
    {
        using ymod_webserver::transformer;
        using ymod_webserver::argument;
        auto webserver = yplatform::find<ymod_webserver::server>("web_server");
        webserver->bind(
            "", { "/ping" }, io_->wrap(weak_bind(&module::ping, shared_from(this), ph::_1)));
        webserver->bind(
            "",
            { "/raw_get" },
            io_->wrap(std::bind(&module::raw_get, shared_from(this), ph::_1, ph::_2)),
            transformer(argument<string>("key")));
        webserver->bind(
            "",
            { "/raw_set" },
            io_->wrap(
                weak_bind(&module::raw_set, shared_from(this), ph::_1, ph::_2, ph::_3, ph::_4)),
            transformer(
                argument<string>("key"), argument<string>("value"), argument<int32_t>("ttl")));
        webserver->bind(
            "",
            { "/get" },
            io_->wrap(weak_bind(&module::get, shared_from(this), ph::_1, ph::_2)),
            transformer(argument<string>("key")));
        webserver->bind(
            "",
            { "/set" },
            io_->wrap(weak_bind(&module::set, shared_from(this), ph::_1, ph::_2, ph::_3, ph::_4)),
            transformer(
                argument<string>("key"), argument<string>("value"), argument<int32_t>("ttl")));
    }

private:
    using stream_ptr = ymod_webserver::http::stream_ptr;
    using clock = yplatform::time_traits::clock;
    using nodes_vector = std::vector<std::shared_ptr<yhttp::cluster_client>>;
    using response = yhttp::response;

    struct get_op
    {
        get_op(stream_ptr stream, const string& key, std::shared_ptr<module> module)
            : stream(stream), key(key), module(module)
        {
        }

        void operator()(yield_context<get_op> yield_ctx, error_code err = {}, response resp = {})
        {
            try
            {
                reenter(yield_ctx)
                {
                    if (entry = module->cache_.get(key))
                    {
                        stream->result(ymod_webserver::codes::ok, entry->to_string());
                        yield break;
                    }
                    for (it = module->nodes_.begin(); it != module->nodes_.end(); ++it)
                    {
                        yield(*it)->async_run(
                            stream->ctx(), make_request(), module->io_->wrap(yield_ctx));
                        if (err)
                        {
                            YLOG_CTX(module->logger(), stream->ctx(), error)
                                << "get_op error: " << err.message();
                        }
                        else if (resp.status == 200)
                        {
                            stream->result(ymod_webserver::codes::ok, resp.body);
                            yield break;
                        }
                    }
                    stream->result(ymod_webserver::codes::no_content);
                }
            }
            catch (const std::exception& e)
            {
                YLOG_CTX(module->logger(), stream->ctx(), error)
                    << "get_op exception: " << e.what();
                stream->result(ymod_webserver::codes::internal_server_error);
            }
        }

        yhttp::request make_request()
        {
            return yhttp::request::POST("/raw_get", yhttp::url_encode({ { "key", key } }, 0));
        }

        stream_ptr stream;
        string key;
        std::shared_ptr<module> module;
        nodes_vector::iterator it;
        optional<cache_entry> entry;
    };

    struct set_op
    {
        set_op(
            stream_ptr stream,
            const string& key,
            const cache_entry& entry,
            std::shared_ptr<module> module)
            : stream(stream), key(key), entry(entry), module(module)
        {
        }

        void operator()(yield_context<set_op> yield_ctx, error_code err = {}, response = {})
        {
            try
            {
                reenter(yield_ctx)
                {
                    module->cache_.set(key, entry);
                    for (it = module->nodes_.begin(); it != module->nodes_.end(); ++it)
                    {
                        yield(*it)->async_run(
                            stream->ctx(), make_request(), module->io_->wrap(yield_ctx));
                        if (err)
                        {
                            YLOG_CTX(module->logger(), stream->ctx(), error)
                                << "set_op error: " << err.message();
                        }
                    }
                    stream->result(ymod_webserver::codes::ok);
                }
            }
            catch (const std::exception& e)
            {
                YLOG_CTX(module->logger(), stream->ctx(), error)
                    << "set_op exception: " << e.what();
                stream->result(ymod_webserver::codes::internal_server_error);
            }
        }

        yhttp::request make_request()
        {
            return yhttp::request::POST(
                "/raw_set",
                yhttp::url_encode(
                    { { "key", key },
                      { "value", entry.value },
                      { "ttl", std::to_string(entry.ttl()) } },
                    0));
        }

        stream_ptr stream;
        string key;
        cache_entry entry;
        std::shared_ptr<module> module;
        nodes_vector::iterator it;
    };

    void ping(stream_ptr stream)
    {
        stream->result(ymod_webserver::codes::ok, "pong");
    }

    void get(stream_ptr stream, const string& key)
    {
        yplatform::spawn(std::make_shared<get_op>(stream, key, shared_from(this)));
    }

    void set(stream_ptr stream, const string& key, const string& value, int32_t ttl)
    {
        yplatform::spawn(std::make_shared<set_op>(
            stream, key, cache_entry(value, time_traits::seconds(ttl)), shared_from(this)));
    }

    void raw_get(stream_ptr stream, const string& key)
    {
        if (auto entry = cache_.get(key))
        {
            stream->result(ymod_webserver::codes::ok, entry->to_string());
        }
        else
        {
            stream->result(ymod_webserver::codes::no_content);
        }
    }

    void raw_set(stream_ptr stream, const string& key, const string& value, int32_t ttl)
    {
        cache_.set(key, cache_entry(value, time_traits::seconds(ttl)));
        stream->result(ymod_webserver::codes::ok);
    }

    boost::asio::io_service* io_;
    local_cache cache_;
    nodes_vector nodes_;
};

}

REGISTER_MODULE(rcache::module)

#include <yplatform/unyield.h>