import os
import yaml

from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.yhttp_service import YplatformHttpService
from mail.devpack.lib.components.base import BaseComponent
from mail.webmail_config.lib.make_config import make_config


class RetrieverTesting(BaseComponent):
    NAME = 'retriever_testing'
    DEPS = []

    @staticmethod
    def gen_config(port_generator, config=None):
        return YplatformHttpService.gen_config(port_generator, config=config)

    def __init__(self, env, components):
        super(RetrieverTesting, self).__init__(env, components)
        self.__service = YplatformHttpService(
            env=env,
            name=self.NAME,
            binary_name='retriever',
            custom_path='retriever'
        )

    def _init_root(self):
        self._generate_config('testing')
        self._generate_config_devpack()

    def _generate_config(self, env):
        base = resource.find('retriever/base.yml')
        service_yaml = yaml.safe_load(resource.find('retriever/service.yaml'))
        common_yaml = yaml.safe_load(resource.find('webmail_config/common.yaml'))
        cfg = make_config(env, base, service_yaml, common_yaml, silent=True)
        config = self.__service.format_config(cfg)
        helpers.write2file(config, os.path.join(self.etc_path, 'retriever.yml'))

    def _generate_config_devpack(self):
        devpack = self.__service.format_config(resource.find('retriever/testing.yml'))
        helpers.write2file(devpack, os.path.join(self.etc_path, 'retriever-devpack.yml'))

    def _copy_files(self):
        helpers.write2file(resource.find('retriever/tvm_secret'), os.path.join(self.etc_path, 'tvm_secret'))
        helpers.write2file(resource.find('retriever/mulcagate_url_key.txt'), os.path.join(self.etc_path, 'mulcagate_url_key.txt'))

        helpers.write2file(resource.find('content_type_detector_config/libmagic.issues'), os.path.join(self.etc_path, 'libmagic.issues'))
        helpers.write2file(resource.find('content_type_detector_config/magic'),           os.path.join(self.etc_path, 'magic'))
        helpers.write2file(resource.find('content_type_detector_config/mime.types'),      os.path.join(self.etc_path, 'mime.types'))
        helpers.write2file(resource.find('content_type_detector_config/mime-types.xml'),  os.path.join(self.etc_path, 'mime-types.xml'))

        recognizer_path = os.path.join(self.get_root(), 'usr', 'share', 'recognizer')
        helpers.mkdir_recursive(recognizer_path)
        helpers.write2file(resource.find('recognizer/dict.dict'),        os.path.join(recognizer_path, 'dict.dict'))
        helpers.write2file(resource.find('recognizer/queryrec.dict'),    os.path.join(recognizer_path, 'queryrec.dict'))
        helpers.write2file(resource.find('recognizer/queryrec.weights'), os.path.join(recognizer_path, 'queryrec.weights'))

        uatraits_path = os.path.join(self.get_root(), 'usr', 'share', 'uatraits')
        helpers.mkdir_recursive(uatraits_path)
        helpers.write2file(resource.find('uatraits/browser.xml'),     os.path.join(uatraits_path, 'browser.xml'))
        helpers.write2file(resource.find('uatraits/profiles.xml'),    os.path.join(uatraits_path, 'profiles.xml'))

        helpers.write2file(resource.find('attach_sid_aes_keys.json'), os.path.join(self.etc_path, 'attach_sid_aes_keys.json'))
        helpers.write2file(resource.find('attach_sid_hmac_keys.json'), os.path.join(self.etc_path, 'attach_sid_hmac_keys.json'))

    @property
    def etc_path(self):
        return self.__service.get_etc_path()

    def init_root(self):
        self.__service.init_root()
        self._init_root()
        self._copy_files()

    def start(self):
        self.__service.start('pong')

    def stop(self):
        self.__service.stop()

    def info(self):
        res = self.__service.info()
        res.update({'state': self.state})
        return res

    def prepare_data(self):
        pass

    def get_root(self):
        return self.__service.get_root()

    def webserver_port(self):
        return self.__service.webserver_port
