#include <src/logic/message_part_real/make_attach.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <boost/range/algorithm/find_if.hpp>
#include <tuple>

namespace retriever {
namespace {

class MessageAccessPool {
public:
    MessageAccessPool(StorageServicePtr storageService, MakeMessageAccess makeMessageAccess,
                      const Recognizer::Wrapper& recognizer)
            : storageService(std::move(storageService)),
              makeMessageAccess(std::move(makeMessageAccess)),
              recognizer(recognizer) {
    }

    MessageAccessPtr get(Stid stid, macs::MimeParts mimeParts, YieldContext yc) {
        auto it = cache.find(stid);
        if (it == cache.end()) {
            auto stidCopy = stid;
            it = cache.insert({std::move(stid),
                makeMessageAccess(std::move(stidCopy), std::move(mimeParts), storageService, recognizer, yc)}).first;
        }
        return it->second;
    }

private:
    StorageServicePtr storageService;
    MakeMessageAccess makeMessageAccess;
    const Recognizer::Wrapper& recognizer;
    std::unordered_map<Stid, MessageAccessPtr> cache;
};

template <class GetMessageAccess>
MessagePart getMessagePartByHid(const Hid& hid, const hound::MessageParts& messageParts,
        GetMessageAccess getMessageAccess) {
    boost::optional<macs::MimePart> mimePart;
    Stid stid;
    Hid stidHid;
    macs::MimeParts allMimeParts;
    const auto otherMessagePart = boost::range::find_if(messageParts.other,
        [&] (const auto& v) { return v.second.count(hid); });
    if (otherMessagePart != messageParts.other.end()) {
        const auto& messagePart = otherMessagePart->second.at(hid);
        stid = otherMessagePart->first;
        stidHid = messagePart.stidHid;
        mimePart = macs::MimePartFactory(messagePart.mimePart).hid(stidHid).release();
        allMimeParts.push_back(mimePart.get());
    } else {
        stid = messageParts.root.stid;
        stidHid = hid;
        const auto rootMessagePart = messageParts.root.mimeParts.find(hid);
        if (rootMessagePart != messageParts.root.mimeParts.end()) {
            mimePart = rootMessagePart->second;
        }
        boost::transform(messageParts.root.mimeParts, std::back_inserter(allMimeParts),
            [] (const auto& v) { return v.second; });
    }
    const auto messageAccess = getMessageAccess(stid, std::move(allMimeParts));
    return makeMessagePart(std::move(stid), messageAccess->getBody(stidHid),
            mimePart.is_initialized() ? mimePart.get() : messageAccess->getHeaderStruct(stidHid));
}

} // namespace

MessageAccessPtr makeDefaultMessageAccess(Stid stid, macs::MimeParts mimeParts, StorageServicePtr storageService, const Recognizer::Wrapper& recognizer, YieldContext yc) {
    return storageService->createMessageAccess(mail_getter::getMessageAccessParams(std::move(stid), std::move(mimeParts)), recognizer, yc);
}

MakeAttach::MakeAttach(TaskContextPtr context, StorageServicePtr storageService, HoundClientPtr houndClient,
        const Recognizer::Wrapper& recognizer, boost::optional<std::string> zipArchiveEncoding,
        MakeMessageAccess makeMessageAccess)
        : context(std::move(context)),
          storageService(std::move(storageService)),
          houndClient(std::move(houndClient)),
          recognizer(recognizer),
          zipArchiveEncoding(std::move(zipArchiveEncoding)),
          makeMessageAccess(std::move(makeMessageAccess)) {
}

Attach MakeAttach::operator ()(const mail_getter::part_id::Old& partId) const {
    const auto messageAccess = makeMessageAccess(partId.stid, macs::MimeParts(), storageService, recognizer,
                                                 context->yieldContext());
    if (zipArchiveEncoding.is_initialized()) {
        const auto hids = parseMultiHid(partId.hid);
        std::vector<Attach> messageParts;
        messageParts.reserve(hids.size());
        boost::range::transform(hids, std::back_inserter(messageParts),
            [&] (const auto& hid) {
                return makeMessagePart(partId.stid, messageAccess->getBody(hid),
                        messageAccess->getHeaderStruct(hid));
            });
        auto zipArchive = create_zip_archive(zipArchiveEncoding.get(), messageParts);
        Attach result;
        result.content = std::move(zipArchive.content);
        result.type = std::move(zipArchive.type);
        result.subtype = std::move(zipArchive.subtype);
        return result;
    } else {
        return makeMessagePart(partId.stid, messageAccess->getBody(partId.hid),
                messageAccess->getHeaderStruct(partId.hid));
    }
}

Attach MakeAttach::operator ()(const mail_getter::part_id::Temporary& partId) const {
    const auto messageAccess = makeMessageAccess(partId.stid, macs::MimeParts(), storageService, recognizer, context->yieldContext());
    return makeMessagePart(partId.stid, messageAccess->getBody(partId.hid),
            messageAccess->getHeaderStruct(partId.hid));
}

Attach MakeAttach::operator ()(const mail_getter::part_id::SingleMessagePart& partId) const {
    const auto messageParts = getMessageParts(partId.uid, partId.mid);
    return getMessagePartByHid(partId.hid, messageParts,
        [&] (auto stid, auto mimeParts) {
            return makeMessageAccess(std::move(stid), std::move(mimeParts), storageService, recognizer, context->yieldContext());
        });
}

Attach MakeAttach::operator ()(const mail_getter::part_id::MultipleMessagePart& partId) const {
    using mail_getter::part_id::SingleMessagePart;
    if (!zipArchiveEncoding.is_initialized()) {
        std::ostringstream stream;
        stream << "zipArchiveEncoding is not initialized in " << __PRETTY_FUNCTION__ << " at " << __FILE__ << ":" << __LINE__;
        throw std::invalid_argument(stream.str());
    }
    const auto messageParts = getMessageParts(partId.uid, partId.mid);
    MessageAccessPool messageAccessPool(storageService, makeMessageAccess, recognizer);
    std::vector<MessagePart> files;
    files.reserve(partId.hids.size());
    boost::range::transform(partId.hids, std::back_inserter(files),
        [&] (const auto& hid) {
            return getMessagePartByHid(hid, messageParts,
                [&] (auto stid, auto mimeParts) {
                    return messageAccessPool.get(std::move(stid), std::move(mimeParts), context->yieldContext());
                });
        });
    auto zipArchive = create_zip_archive(zipArchiveEncoding.get(), files);
    Attach result;
    result.content = std::move(zipArchive.content);
    result.type = std::move(zipArchive.type);
    result.subtype = std::move(zipArchive.subtype);
    return result;
}

hound::MessageParts MakeAttach::getMessageParts(const Uid& uid, const Mid& mid) const {
    auto messageParts = houndClient->getMessageParts(context, uid, mid);
    if (!messageParts.is_initialized()) {
        throw NoMessageParts("no message parts for uid=" + uid + " mid=" + mid);
    }
    return std::move(messageParts.get());
}

} // namespace retriever
