#include <src/logic/message_part_real/message_part.hpp>
#include <src/logic/message_part_real/ua.hpp>
#include <butil/StrUtils/Encoding.h>
#include <butil/StrUtils/Iconv.h>
#include <butil/StrUtils/utfutils.h>
#include <butil/butil.h>
#include <butil/digest.h>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/lexical_cast.hpp>
#include <zipios++/zipoutputstream.h>
#include <set>
#include <zlib.h>

namespace retriever {

MessagePart makeMessagePart(std::string stid, std::string body, const MetaPart& attributes) {
    MessagePart result;

    Encoding::decode(attributes.encoding(), body, true);
    result.content = std::move(body);

    result.filename = !attributes.fileName().empty() ? attributes.fileName() : attributes.name();

    const auto &chr = attributes.charset();
    result.charset = chr.empty() ? "us-ascii" : chr;

    const auto &contentType = attributes.contentType();
    const auto &contentSubType = attributes.contentSubtype();

    if (contentType == "message" && contentSubType == "delivery-status") {
        result.type = "text";
        result.subtype = "plain";
    } else {
        result.type = contentType;
        result.subtype = contentSubType;
    }

    utf::to_lower(result.type);
    utf::to_lower(result.subtype);

    if (result.type == "image" && result.subtype == "pjpeg") {
        result.subtype = "jpeg"; // especially for IE
    }

    result.stid = std::move(stid);
    result.offsetBegin = attributes.offsetBegin();
    result.offsetEnd = attributes.offsetEnd();

    return result;
}

std::string make_content_type(const std::string& filename,
                              const std::string& type,
                              const std::string& subtype,
                              const std::string& charset)
{
    std::string result = type + "/" + subtype;
    if(!filename.empty()) result += ";filename=\"" + UA::encodeUrlControlChars(filename) + "\"";
    if(type == "text") result += ";charset=\"" + charset+ "\"";
    return result;
}


class NonameGenerator {
public:
    NonameGenerator()
        : counter(1)
    {}
    void processFilename(std::string& filename) {
        if(filename.empty()) {
            filename = "NoName-" + boost::lexical_cast<std::string>(counter++);
        } else if (filenames.count(filename)) {
            std::string::size_type dotPos = filename.rfind(".");
            const std::string name = filename.substr(0, dotPos);
            const std::string dotExt = (dotPos == std::string::npos) ? "" : filename.substr(dotPos);
            filename = name + "-" + boost::lexical_cast<std::string>(counter++) + dotExt;
        }
        filenames.insert(filename);
    }
private:
    int counter;
    std::set<std::string> filenames;
};

unsigned crc32(const std::string& str) {
    const unsigned char* data = reinterpret_cast<const unsigned char*>(str.c_str());
    return unsigned(::crc32(0, data, uInt(str.length())));
}

std::vector<unsigned char> make_zip_utf_filename(const std::string& filename,
                                                 const std::string& dosfilename)
{
    const auto length = filename.length() + 5;
    const unsigned int crc = crc32(dosfilename);
    std::vector<unsigned char> result(length + 4);
    result[0] = 0x75;
    result[1] = 0x70;
    result[2] = static_cast<unsigned char>(length & 0xff);
    result[3] = static_cast<unsigned char>(length >> 8);
    result[4] = 1;
    result[5] = crc & 0xff;
    result[6] = (crc >> 8) & 0xff;
    result[7] = (crc >> 16) & 0xff;
    result[8] = static_cast<unsigned char>(crc >> 24);
    for(size_t i = 0; i < filename.length(); ++i)
        result[i + 9] = static_cast<unsigned char>(filename[i]);
    return result;
}

void put_file_in_zip(zipios::ZipOutputStream& os, MessagePart messagePart,
                     const std::string& encoding, Iconv& cnv,
                     NonameGenerator& nonamer)
{
    nonamer.processFilename(messagePart.filename);

    if(encoding == "utf-8") {
        os.putNextEntry(messagePart.filename);
    } else  {
        std::string dosfilename = cnv.recode(messagePart.filename, '_');
        zipios::ZipCDirEntry entry(dosfilename, "",
                                   make_zip_utf_filename(messagePart.filename, dosfilename),
                                   zipios::WriterVersion::dos_version);
        os.putNextEntry(entry);
    }
    os.setMethod(zipios::DEFLATED);
    os.write(messagePart.content.data(), static_cast<std::streamsize>(messagePart.content.size()));
    os.closeEntry();
}

std::vector<std::string> parseMultiHid(const std::string& multiHid) {
    std::vector<std::string> result;
    boost::split(result, multiHid, boost::is_any_of(","));
    return result;
}

ZipArchive create_zip_archive(const std::string& encoding, std::vector<MessagePart> messageParts) {
    NonameGenerator nonamer;

    Iconv cnv ("utf-8", encoding.c_str());
    std::stringstream result;
    zipios::ZipOutputStream os(result);
    for (auto& part : messageParts) {
        put_file_in_zip(os, std::move(part), encoding, cnv, nonamer);
    }
    os.finish();
    return ZipArchive {result.str(), "application", "zip"};
}

} // namespace retriever
