#pragma once

#include <boost/system/error_code.hpp>

namespace retriever {

enum class MessagePartError {
    emptySid = 0,
    invalidSid,
    oldSid,
    illFormedSid,
    expiredSid,
    invalidAesKeyId,
    invalidHmacKeyId,
    decryptionError,
    notFound,
    internal,
};


class MessagePartErrorCategory : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "MessagePartErrorCategory";
    }

    std::string message(int ev) const noexcept override {
        using Err = MessagePartError;
        switch (Err(ev)) {
            case Err::emptySid:
                return "empty sid";
            case Err::invalidSid:
                return "invalid sid";
            case Err::oldSid:
                return "old sid";
            case Err::illFormedSid:
                return "ill formed sid";
            case Err::expiredSid:
                return "expired sid";
            case Err::invalidAesKeyId:
                return "invalid AES key id";
            case Err::invalidHmacKeyId:
                return "invalid HMAC key id";
            case Err::decryptionError:
                return "decryption error";
            case Err::notFound:
                return "not found";
            case Err::internal:
                return "internal error";
        }
        return "message part error";
    }

    static const MessagePartErrorCategory& instance() {
        static MessagePartErrorCategory value;
        return value;
    }
};

inline boost::system::error_code make_error_code(MessagePartError e) {
    return boost::system::error_code(static_cast<int>(e), MessagePartErrorCategory::instance());
}

}

namespace boost::system {

template<>
struct is_error_code_enum<retriever::MessagePartError> : public std::true_type {};

} // namespace boost::system
