#include <src/logic/message_part_real/message_part_real.hpp>
#include <src/logic/message_part_real/make_attach.hpp>
#include <src/logic/message_part_real/resizer.hpp>
#include <src/logic/message_part_real/message_part_errors.hpp>
#include <src/services/mulcagate/logging.hpp>
#include <src/services/mulcagate/http.hpp>

#include <mail/tvm_guard/ymod_tvm/ymod_tvm.h>
#include <mail_getter/AttachShieldCrypto.h>
#include <mail_getter/attach_sid_keys.h>
#include <mail_getter/mime_type.h>

#include <yplatform/find.h>
#include <yplatform/encoding/url_encode.h>

#include <fstream>
#include <stdexcept>

namespace retriever {
namespace {

struct LogPartId : public boost::static_visitor<> {
    LogPartId(const Logger& logger) : logger(logger) {}

    void operator ()(const mail_getter::part_id::Old& partId) const {
        LOGDOG_(logger, notice, log::stid=partId.stid, log::hid=partId.hid, logdog::message="part id is old");
    }

    void operator ()(const mail_getter::part_id::Temporary& partId) const {
        LOGDOG_(logger, notice, log::stid=partId.stid, log::hid=partId.hid, logdog::message="part id is temporary");
    }

    void operator ()(const mail_getter::part_id::SingleMessagePart& partId) const {
        LOGDOG_(logger, notice, logdog::uid=partId.uid, log::mid=partId.mid, log::hid=partId.hid, logdog::message="part id is single message part");
    }

    void operator ()(const mail_getter::part_id::MultipleMessagePart& partId) const {
        LOGDOG_(logger, notice, logdog::uid=partId.uid, log::mid=partId.mid, log::hid=boost::join(partId.hids, ", "), logdog::message="part id is multiple message part");
    }

private:
    const Logger& logger;
};

std::string readKey(const std::string& path) {
    try {
        std::ifstream file;
        file.exceptions(std::ifstream::failbit | std::ifstream::badbit);
        file.open(path);
        std::stringstream result;
        result << file.rdbuf();
        return result.str();
    } catch (const std::exception& e) {
        throw std::runtime_error("Can't read key from file \"" + path + "\": " + e.what());
    }
}

std::string generateImageUrl(const SignedMulcagateUrlGenerator& generator, const Attach& attach) {
    return generator.generate(attach.stid, attach.offsetBegin, attach.offsetEnd - attach.offsetBegin);
}

Content makeContent(Attach&& attach, ResizeClient::Image&& image) {
    Content result;
    result.contentDisposition = std::move(image.contentDisposition);
    result.contentType = image.contentType ? std::move(image.contentType.get())
        : make_content_type(attach.filename, attach.type, attach.subtype, attach.charset);
    result.content = std::move(image.content);
    return result;
}

auto& getExecutor(const YieldContext& yield) {
#if BOOST_VERSION < 106600
    return yield.handler_.dispatcher_.get_io_service();
#else
    using boost::asio::io_context;
    using boost::asio::strand;
    decltype(auto) executor = get_associated_executor(yield.handler_);
    if (auto target = executor.target<strand<io_context::executor_type>>()) {
        return target->get_inner_executor().context();
    } else if (auto target = executor.target<io_context::strand>()) {
        return target->context();
    } else if (auto target = executor.target<io_context::executor_type>()) {
        return target->context();
    }
    throw std::invalid_argument("can't obtain io_context from executor");
#endif
}

} // namespace

MessagePartReal::MessagePartReal(
        AliasClassList mimeTypesList,
        std::unique_ptr<UA> ua,
        std::unique_ptr<ContentTypeDetector> detector,
        mail_getter::ServiceFactoryPtr mailStorage,
        HoundClientPtr houndClient,
        std::size_t maxParallelIoCount,
        Duration maxParallelIoWaitDuration,
        ResizeClientPtr resizeClient,
        SignedMulcagateUrlGenerator signedMulcagateUrlGenerator,
        Recognizer::WrapperPtr recognizer,
        mail_getter::attach_sid::KeyContainer keys)
    : mimeTypesList(std::move(mimeTypesList)),
      ua(std::move(ua)),
      detector(std::move(detector)),
      mailStorage(mailStorage),
      houndClient(houndClient),
      maxParallelIoWaitDuration(maxParallelIoWaitDuration),
      pool(maxParallelIoCount, std::numeric_limits<std::size_t>::max(), std::chrono::seconds(60)),
      resizeClient(resizeClient),
      signedMulcagateUrlGenerator(std::move(signedMulcagateUrlGenerator)),
      recognizer(std::move(recognizer)),
      keys(std::move(keys)) {
}

bool MessagePartReal::canBeBrowsed(const MimeType& mType) const
{
    return mimeTypesList.canBeBrowsed(mType);
}

bool MessagePartReal::canMakeThumbnail(const MimeType& mType)
{
    return mimeTypesList.canBeThumbnailed(mType);
}

bool MessagePartReal::canExifRotate(const MimeType& mType)
{
    return mimeTypesList.canBeRotated(mType);
}

bool MessagePartReal::viewForbidden(const MimeType& mType)
{
    return mimeTypesList.viewForbidden(mType);
}

std::string MessagePartReal::getFilenamesEncoding4Zip(const std::string& userAgent, const std::string& acceptLanguage) const {
    return ua.get() ? ua->getNativeEncoding(userAgent, acceptLanguage) : "cp866";
}

Result MessagePartReal::handle(const TaskContextPtr& context, const Logger& logger, const Parameters& parameters) {
    try {
        if (parameters.sid.empty()) {
            return ErrorCode(MessagePartError::emptySid);
        }

        mail_getter::part_id::Variant partId;

        try {
            mail_getter::attach_sid::Unpacker unpacker(keys);
            partId = unpacker(parameters.sid);
        } catch (const mail_getter::attach_sid::OldSidFormat& e) {
            return ErrorCode(MessagePartError::oldSid, e.what());
        } catch (const mail_getter::attach_sid::IllformedSid& e) {
            return ErrorCode(MessagePartError::illFormedSid, e.what());
        } catch (const mail_getter::attach_sid::ExpiredSid& e) {
            return ErrorCode(MessagePartError::expiredSid, e.what());
        } catch (const mail_getter::attach_sid::InvalidAesKeyId& e) {
            return ErrorCode(MessagePartError::invalidAesKeyId, e.what());
        } catch (const mail_getter::attach_sid::InvalidHmacKeyId& e) {
            return ErrorCode(MessagePartError::invalidHmacKeyId, e.what());
        } catch (const mail_getter::attach_sid::DecryptionError& e) {
            return ErrorCode(MessagePartError::decryptionError, e.what());
        } catch (const std::runtime_error& e) {
            return ErrorCode(MessagePartError::invalidSid, e.what());
        }

        boost::apply_visitor(LogPartId(logger), partId);

        Attach attach;
        boost::optional<std::string> zipArchiveEncoding;
        if (parameters.archive == "zip" || partId.type() == typeid(mail_getter::part_id::MultipleMessagePart)) {
            zipArchiveEncoding = getFilenamesEncoding4Zip(parameters.userAgent, parameters.acceptLanguage);
        }

        boost::system::error_code ec;
        const auto handle = pool.get_auto_recycle(
            getExecutor(context->yieldContext()),
            context->yieldContext()[ec],
            maxParallelIoWaitDuration
        );

        if (ec) {
            return ErrorCode(MessagePartError::internal, "Get io permission error: " + ec.message());
        }

        const auto mgLogger = std::make_shared<MulcagateLog>(logger);
        try {
            MakeAttach makeAttach(context,
                mailStorage->createService(mgLogger, tvm::Ticket(), context->requestId()),
                houndClient,
                *recognizer,
                zipArchiveEncoding);
            attach = boost::apply_visitor(makeAttach, partId);
        } catch (const NoMessageParts& e) {
            return ErrorCode(MessagePartError::notFound, e.what());
        } catch (const mail_getter::MessageNotFound& e) {
            return ErrorCode(MessagePartError::notFound, e.what());
        }

        if (attach.filename.empty()) {
            attach.filename = parameters.name;
        }

        MimeType mType;
        MimeType saveMimeType;
        try {
            mType = detector->detect( attach.filename, attach.content );
            LOGDOG_(logger, debug, logdog::message="MIME-type detected by file name: " + mType.toString());
            saveMimeType  = detector->detectByContent( attach.content );
            LOGDOG_(logger, debug, logdog::message="MIME-type detected by file content: " + saveMimeType.toString());
        } catch (const std::runtime_error & e) {
            saveMimeType = mType = detector->detectByFilename(attach.filename);
            LOGDOG_(logger, notice, logdog::message="Exception detecting by file content", logdog::exception=e);
        }
        ContentTypeDetector::adjustType(mType);
        if (!mType.isDefaultMimeType()) {
            LOGDOG_(logger, debug, logdog::message="Attach " + attach.filename + " has nontrivial mime type: " + mType.toString());
            attach.type = mType.type();
            attach.subtype = mType.subtype();
        }

        const bool needResize = parameters.thumbnailSize || parameters.maxSize;
        const auto convertFromBase64 = partId.type() != typeid(mail_getter::part_id::Temporary);
        boost::optional<Resizer> resizer;

        if (needResize && canMakeThumbnail(saveMimeType)) {
            if (parameters.thumbnailSize) {
                LOGDOG_(logger, debug, logdog::message="Attach will be converted to thumbnail");
                resizer = Resizer();
                resizer.get()
                    .cropRect(true)
                    .exifRotate(parameters.exifRotate)
                    .size(parameters.thumbnailSize.get())
                    .fromBase64(convertFromBase64);
            } else {
                LOGDOG_(logger, debug, logdog::message="Attach will be resized");
                resizer = Resizer();
                resizer.get()
                    .exifRotate(parameters.exifRotate)
                    .size(parameters.maxSize.get())
                    .fromBase64(convertFromBase64);
            }
        } else if (parameters.exifRotate && canExifRotate(saveMimeType)) {
            LOGDOG_(logger, debug, logdog::message="Attach will rotated by EXIF");
            resizer = Resizer();
            if (!parameters.noDisposition || viewForbidden(saveMimeType)) {
                resizer.get().attachment(attach.filename);
            }
            resizer.get()
                .exifRotate(true)
                .fromBase64(convertFromBase64);
        }

        if (resizer) {
            const auto imageUrl = generateImageUrl(signedMulcagateUrlGenerator, attach);
            auto url = resizer->genurl(context, imageUrl, *resizeClient);
            if (auto image = resizeClient->get(context, std::move(url))) {
                return makeContent(std::move(attach), std::move(*image));
            }
            LOGDOG_(logger, notice, logdog::message="Failed to resize image with MIME-type detected by content: " + saveMimeType.toString());
        }

        LOGDOG_(logger, debug, logdog::message="Attach will be returned as is");
        Content result;
        if (!parameters.noDisposition || viewForbidden(mType) || viewForbidden(saveMimeType)) {
            result.contentDisposition = "attachment";
            if (!attach.filename.empty()) {
                result.contentDisposition.get() += ";"
                    + ua->filenameForBrowser(attach.filename, parameters.userAgent,
                        parameters.wapProfile, parameters.operaminiPhoneUa);
            }
        }
        result.contentType = make_content_type(attach.filename, attach.type, attach.subtype, attach.charset);
        result.content = std::move(attach.content);
        return result;
    } catch (const std::exception& e) {
        return ErrorCode(MessagePartError::internal, e.what());
    }
}

MessagePartReal::Stats MessagePartReal::getStats() const {
    MessagePartReal::Stats result;
    result.pool = pool.stats();
    return result;
}

std::shared_ptr<MessagePartReal> makeMessagePartReal(const Config& config) {
    const auto tvm2Module = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>(config.tvm_module);

    if (!tvm2Module) {
        LOGDOG_(makeLoggerWithRequestId(""), error, logdog::message="cannot get tvm2 module");
        throw std::runtime_error("cannot get tvm2 module");
    }

    return std::make_shared<MessagePartReal>(
        AliasClassList(config.mime_aliases),
        std::make_unique<UA>(config.uatraits_config, config.uatraits_profiles_config),
        std::make_unique<ContentTypeDetector>(config.mime_types, config.libmagic_issues, config.magic_file),
        mail_getter::createServiceFactory(
            config.mulcagate.settings,
            std::make_shared<MulcagateHttpClient>(
                makeGetClusterClient(config.mulcagate.cluster_client_module),
                makeGetServiceTicket(tvm2Module)
            )
        ),
        std::make_shared<const HoundClientImpl>(
            makeGetHttpClient(config.hound.http_client_module),
            makeGetServiceTicket(tvm2Module),
            HoundClientImpl::Config {config.hound.location, config.hound.timeouts,
                                     config.hound.retries, config.hound.service_name}
        ),
        config.max_parallel_io_count,
        std::chrono::duration_cast<MessagePartReal::Duration>(config.max_parallel_io_wait_duration),
        std::make_shared<const ResizeClientImpl>(
            makeGetClusterClient(config.resize.genurl.cluster_client_module),
            makeGetHttpClient(config.resize.get.http_client_module),
            makeGetServiceTicket(tvm2Module),
            ResizeClientImpl::Config {config.resize.genurl, config.resize.get}
        ),
        SignedMulcagateUrlGenerator(
            config.mulcagate_url.host + ":" + std::to_string(config.mulcagate_url.port),
            config.mulcagate.settings->service,
            readKey(config.mulcagate_url.key_path),
            config.mulcagate_url.ttl
        ),
        Recognizer::create(
            config.recognizer.languageDictPath.c_str(),
            config.recognizer.languageWeightsPath.c_str(),
            config.recognizer.encodingDictPath.c_str()
        ),
        mail_getter::attach_sid::parseKeyContainer(
                config.attach_sid_keys.aesKeysPath,
                config.attach_sid_keys.hmacKeysPath
        )
    );
}

} // namespace retriever
