#pragma once

#include <src/log.hpp>
#include <src/config.hpp>
#include <src/logic/errors.hpp>
#include <src/logic/message_part_real/ua.hpp>
#include <src/logic/message_part_real/image_size.hpp>
#include <src/logic/message_part_real/signed_mulcagate_url_generator.hpp>

#include <mail_getter/alias_class_list.h>
#include <mail_getter/content_type.h>
#include <mail_getter/service_factory.h>

#include <mail_errors/error_code.h>

#include <yamail/resource_pool/async/pool.hpp>

#include <boost/property_tree/ptree.hpp>

namespace retriever {

struct Parameters {
    std::string name;
    std::string sid;
    bool noDisposition;
    bool exifRotate;
    std::string archive;
    std::string userAgent;
    std::string acceptLanguage;
    std::string wapProfile;
    std::string operaminiPhoneUa;
    boost::optional<ImageSize> thumbnailSize;
    boost::optional<ImageSize> maxSize;
};

struct Content {
    std::string contentType;
    boost::optional<std::string> contentDisposition;
    std::string content;
};

using ErrorCode = mail_errors::error_code;
using Result = boost::variant<Content, ErrorCode>;

class MessagePartReal {
public:
    using ptree = boost::property_tree::ptree;
    using Duration = yamail::resource_pool::time_traits::duration;

    struct Stats {
        yamail::resource_pool::async::stats pool;
    };

    MessagePartReal(
            AliasClassList mimeTypesList,
            std::unique_ptr<UA> ua,
            std::unique_ptr<ContentTypeDetector> detector,
            mail_getter::ServiceFactoryPtr mailStorage,
            HoundClientPtr houndClient,
            std::size_t maxParallelIoCount,
            Duration maxParallelIoWaitDuration,
            ResizeClientPtr resizeClient,
            SignedMulcagateUrlGenerator signedMulcagateUrlGenerator,
            Recognizer::WrapperPtr recognizer,
            mail_getter::attach_sid::KeyContainer keys
    );

    Result handle(const TaskContextPtr& context, const Logger& logger, const Parameters& parameters);

    Stats getStats() const;

private:
    struct IoPermission {};

    using IoPermissionsPool = yamail::resource_pool::async::pool<IoPermission>;

    bool canBeBrowsed(const MimeType& mType) const;
    bool canMakeThumbnail(const MimeType& mType);
    bool canExifRotate(const MimeType& mType);
    bool viewForbidden(const MimeType& mType);
    std::string getFilenamesEncoding4Zip(const std::string& userAgent, const std::string& acceptLanguage) const;

    AliasClassList mimeTypesList;
    std::unique_ptr<UA> ua;
    std::unique_ptr<ContentTypeDetector> detector;
    mail_getter::ServiceFactoryPtr mailStorage;
    HoundClientPtr houndClient;
    Duration maxParallelIoWaitDuration;
    IoPermissionsPool pool;
    ResizeClientPtr resizeClient;
    SignedMulcagateUrlGenerator signedMulcagateUrlGenerator;
    Recognizer::WrapperPtr recognizer;
    ::mail_getter::attach_sid::KeyContainer keys;
};

std::shared_ptr<MessagePartReal> makeMessagePartReal(const Config& config);

} // namespace retriever
