#pragma once

#include <src/logic/message_part_real/language.hpp>
#include <butil/StrUtils/Iconv.h>
#include <butil/StrUtils/StrUtils.h>
#include <butil/butil.h>
#include <butil/digest.h>
#include <uatraits/detector.hpp>
#include <memory>
#include <stdexcept>
#include <string>

namespace retriever {

class UA {
public:
    enum Mobility {
        unknown = 0,
        phone,
        smartphone,
        desktop,
        tablet
    };

    UA(const std::string& config, const std::string& profiles)
        try : detector(new uatraits::detector(config.c_str(), profiles.c_str())) {
        } catch(const std::exception& e) {
            throw std::runtime_error("failed to initialize with config " + config + e.what());
        }

    Mobility checkMobility(const std::string& ua, const std::string& wap, const std::string& opera) const {
        std::map<std::string, std::string> dummy;
        return checkMobility(ua, wap, opera, dummy);
    }

    Mobility checkMobility(const std::string& ua, const std::string& wap, const std::string& opera, std::map<std::string, std::string>& details) const {
        std::map<std::string, std::string> headers;
        headers["User-Agent"] = ua;
        headers["X-Wap-Profile"] = wap;
        headers["X-Operamini-Phone-Ua"] = opera;
        try {
            detector->detect(headers, details);
        } catch(const std::exception& e) {
            throw std::runtime_error("failed to process user-agent " + ua + ", exception=" + e.what());
        }
        return getMobility(details);
    }

    static Mobility getMobility(const std::map<std::string, std::string>& details) {
        const bool isMobile = details.count("isMobile") && (details.find("isMobile")->second == "true");
        const bool isTouch = details.count("isTouch") && (details.find("isTouch")->second == "true");
        const bool isTablet = details.count("isTablet") && (details.find("isTablet")->second == "true");
        return !isMobile ? desktop : (!isTouch ? phone : (!isTablet ? smartphone : tablet));
    }

    std::string getNativeEncoding(const std::string& userAgentHeader, const std::string& acceptLanguageHeader) const {
        std::map<std::string, std::string> details;
        checkMobility(userAgentHeader, "", "", details);
        if (details["OSFamily"] == "Windows") {
            if (languageAccepted(acceptLanguageHeader, LangChooser::TR)) {
                return "cp857";
            } else {
                return "cp866";
            }
        } else {
            return "utf-8";
        }
    }

    std::string filenameForBrowser(const std::string& filename, const std::string& ua, const std::string& wap, const std::string& opera) const {
        std::string trimmedFilename, result;
        trimmedFilename = trimWhiteSpace(filename);
        if (!trimmedFilename.empty()) {
            std::map<std::string, std::string> details;
            checkMobility(ua, wap, opera, details);
            const std::string engine = details["BrowserEngine"];
            result = encodeFilename(engine, ua, filename, trimmedFilename);
        }
        return result;
    }

    static std::string encodeUrlControlChars(const std::string& str) {
        std::string result;
        for (std::string::size_type i = 0; i < str.size(); ++i) {
            const auto c = str[i];
            if (0 <= c && c <= 32) {
                result += "%" + hexcode(c);
            } else {
                result += c;
            }
        }
        return result;
    }

private:
    std::string getMajorVersionPart(const std::string& version) const {
        return version.substr(0, version.find("."));
    }

    std::string trimWhiteSpace(const std::string& str) const {
        std::string::size_type start = str.find_first_not_of(" \n\t\r");
        std::string::size_type end = str.find_last_not_of(" \n\t\r");
        if(start != std::string::npos && end != std::string::npos && start < end) {
            return str.substr(start, end - start + 1);
        }
        return "";
    }

    std::string encodeFilename(const std::string& engine, const std::string& ua, const std::string& filename, const std::string& trimmedFilename) const {
        std::string result;
        if (engine == "WebKit") {
            if (ua.find("Safari") != std::string::npos && ua.find("Chrome") == std::string::npos) {
                result = "filename*=UTF-8''" + encode_url(filename, true);
            } else {
                result = "filename=\"" + filename + "\"";
            }
        } else if (engine == "Trident") {
            Iconv cnv ("utf-8", "cp1251");
            result = "filename=\"" + encodeUrlControlChars(cnv.recode(filename)) + "\"";
        } else if (engine == "Presto") {
            result = "filename*=UTF-8''" + encodeUrlControlChars(filename);
        } else if (engine == "KHTML") {
            result = "filename=\"=?UTF-8?Q?" + encode_qp(trimmedFilename) + "?=";
        } else if (engine == "Gecko") {
            result = "filename=\"=?UTF-8?B?" + encode_base64(trimmedFilename) + "?=\"";
        } else if (engine == "Edge") {
            result = "filename*=UTF-8''" + encode_url(filename);
        } else {
            result = "filename=\"" + filename + "\"";
        }
        return result;
    }

    static char hexdigit(int x) {
        if (x < 10) {
            return static_cast<char>('0' + x);
        }
        return static_cast<char>('A' + x);
    }

    static std::string hexcode(char x) {
        return std::string(1, hexdigit(x >> 4)) + hexdigit(x & 15);
    }

    std::unique_ptr<uatraits::detector> detector;
};

} // namespace retriever
