#pragma once

#include <src/config.hpp>
#include <src/server/handlers/message_part_real_handler.hpp>

#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>

#include <yplatform/reactor.h>

namespace retriever {

template <ymod_webserver::methods::http_method method, class Handler>
auto allowOnly(Handler&& handler) {
    using ymod_webserver::http::stream_ptr;
    return [handler = std::forward<Handler>(handler)]
        (const stream_ptr& stream) mutable {
            if (stream->request()->method == method) {
                return handler(stream);
            }
            stream->set_code(ymod_webserver::codes::method_not_allowed);
        };
}

template <class Handler>
auto makeGet(Handler&& handler) {
    return allowOnly<ymod_webserver::methods::mth_get>(std::forward<Handler>(handler));
}

template <class Handler>
auto wrap(std::size_t coroutineStackSize, Handler&& handler) {
    return [handler = std::forward<Handler>(handler), coroutineStackSize]
        (const ymod_webserver::http::stream_ptr& stream) mutable {
            const auto io = yplatform::global_net_reactor->io();
            boost::asio::spawn(*io, [stream, handler]
                (YieldContext yieldContext) mutable {
                    const auto requestId = getOrEmpty(stream->request()->headers, "x-request-id");
                    const auto id = stream->ctx()->uniq_id() + ":" + requestId;
                    const auto context = boost::make_shared<TaskContext>(id, requestId, yieldContext);
                    const auto logger = makeLoggerWithRequestId(id);
                    try {
                        return handler(context, logger, stream);
                    } catch (const boost::coroutines::detail::forced_unwind&) {
                        throw;
                    } catch (const BadMaxSize& e) {
                        LOGDOG_(logger, error, logdog::exception=e);
                        return stream->set_code(ymod_webserver::codes::bad_request);
                    } catch (const boost::system::system_error& e) {
                        LOGDOG_(logger, error, logdog::exception=e);
                    } catch (const std::exception& e) {
                        LOGDOG_(logger, error, logdog::exception=e);
                    } catch (...) {
                        LOGDOG_(logger, error, logdog::message="unknown error");
                    }
                    stream->set_code(ymod_webserver::codes::internal_server_error);
                }, boost::coroutines::attributes(coroutineStackSize));
        };
}

void bindHandlers(ymod_webserver::server& server, const Config& config,
        const std::shared_ptr<MessagePartReal>& messagePartReal) {
    using ymod_webserver::http::stream_ptr;
    server.bind("", {"/ping"}, makeGet([] (const stream_ptr& stream) { stream->result(ymod_webserver::codes::ok, "pong"); }));
    server.bind("", {"message_part_real"}, makeGet(wrap(config.coroutine_stack_size,
        makeMessagePartRealHandler(messagePartReal))));
    server.set_custom_key_extractor("",
        [] (const stream_ptr& stream) {
            const auto& path = stream->request()->url.path;
            if (path.empty()) {
                return std::string();
            }
            if (path.front() == "message_part_real") {
                return path.front();
            }
            return stream->request()->url.make_full_path();
        });
}

} // namespace retriever
