#include <src/server/handlers/message_part_real_handler.hpp>
#include <src/logic/message_part_real/message_part_errors.hpp>

namespace retriever {
namespace {

ymod_webserver::codes::code getHttpCode(MessagePartError code) {
    using Err = MessagePartError;
    switch (code) {
        case Err::emptySid:
            return ymod_webserver::codes::bad_request;
        case Err::oldSid:
        case Err::illFormedSid:
        case Err::expiredSid:
        case Err::invalidAesKeyId:
        case Err::invalidHmacKeyId:
        case Err::invalidSid:
        case Err::notFound:
        case Err::decryptionError:
            return ymod_webserver::codes::unauthorized;
        case Err::internal:
            return ymod_webserver::codes::internal_server_error;
    };
    return ymod_webserver::codes::internal_server_error;
}

boost::optional<ImageSize> getThumbnailSize(const ymod_webserver::param_map_t& params) {
    const auto thumb = params.find("thumb");
    if (thumb == params.end() || thumb->second != "y") {
        return boost::none;
    }
    const auto thumbSize = params.find("thumb_size");
    if (thumbSize == params.end() || thumbSize->second.empty()) {
        return ImageSize {150, 150};
    }
    try {
        const auto result = parseImageSize(thumbSize->second);
        if (result.width == 0 || result.height == 0) {
            return ImageSize {150, 150};
        }
        return result;
    } catch (...) {
        return ImageSize {150, 150};
    }
}

boost::optional<ImageSize> getMaxSize(const ymod_webserver::param_map_t& params) {
    const auto resize = params.find("resize");
    if (resize == params.end() || resize->second != "y") {
        return boost::none;
    }
    const auto maxSize = params.find("max_size");
    if (maxSize == params.end() || maxSize->second.empty()) {
        return boost::none;
    }
    try {
        return parseImageSize(maxSize->second);
    } catch (const std::exception& exception) {
        throw BadMaxSize(exception.what());
    }
}

} // namespace

void Response::operator ()(const Content& value) const {
    stream->set_connection(false);
    stream->set_code(ymod_webserver::codes::ok);
    stream->set_content_type(value.contentType);
    if (value.contentDisposition) {
        stream->add_header("Content-Disposition", value.contentDisposition.get());
    }
    stream->result_chunked()->client_stream() << value.content;
}

void Response::operator ()(const ErrorCode& value) const {
    LOGDOG_(logger, error, logdog::message=value.what(), logdog::error_code=value);
    stream->set_code(getHttpCode(MessagePartError(value.value())));
}

Parameters makeParameters(const ymod_webserver::http::stream& stream) {
    const auto& headers = stream.request()->headers;
    const auto& params = stream.request()->url.params;

    Parameters result;

    result.name = getOrEmpty(params, "name");
    result.sid = getOrEmpty(params, "sid");
    result.noDisposition = getOrEmpty(params, "no_disposition") == "y";
    result.exifRotate = getOrEmpty(params, "exif_rotate") == "y";
    result.archive = getOrEmpty(params, "archive");
    result.userAgent = getOrEmpty(headers, "user-agent");
    result.wapProfile = getOrEmpty(headers, "x-wap-profile");
    result.operaminiPhoneUa = getOrEmpty(headers, "x-operamini-phone-ua");
    result.thumbnailSize = getThumbnailSize(params);
    result.maxSize = getMaxSize(params);

    return result;
}

} // namespace retriever
