#include <src/services/hound/hound_client.hpp>
#include <src/services/hound/hound_error.hpp>
#include <src/services/hound/reflection/error.hpp>
#include <src/services/hound/reflection/mimes.hpp>
#include <yamail/data/deserialization/json_reader.h>
#include <yplatform/encoding/url_encode.h>
#include <http_getter/http_request.h>
#include <mail/http_getter/client/include/fmt.h>
#include <mail/http_getter/client/include/client.h>

#include <boost/fusion/adapted/struct/define_struct.hpp>
#include <src/log.hpp>
#include <src/logic/errors.hpp>

BOOST_FUSION_DEFINE_STRUCT((retriever), HoundMimes,
    (boost::optional<retriever::hound::Mimes>, mimes)
    (boost::optional<retriever::hound::ErrorStruct>, error)
)

namespace retriever {

HoundClientImpl::HoundClientImpl(GetHttpClient getHttpClient, GetServiceTicket getServiceTicket, Config config)
        : getHttpClient(std::move(getHttpClient))
        , getServiceTicket(std::move(getServiceTicket))
        , config(std::move(config))
{ }

template <typename E>
constexpr typename std::underlying_type<E>::type to_underlying(E e) noexcept {
    return static_cast<typename std::underlying_type<E>::type>(e);
}

mail_errors::error_code makeHoundError(HoundError e, const std::string& msg = "") {
    return mail_errors::error_code{to_underlying(e), houndErrorCategory(), msg};
}

OptMessageParts HoundClientImpl::getMessageParts(TaskContextPtr context, const Uid& uid, const Mid& mid) const {
    using yamail::data::deserialization::fromJson;
    using namespace http_getter;

    Request request = get(config.location + "/mimes")
        .headers(
            requestId=context->requestId(),
            serviceTicket=getServiceTicket(config.service_name, context->requestId())
        )
        .getArgs("uid"_arg=uid, "mid"_arg=mid)
        .timeouts(config.timeouts)
        .make();

    const auto httpClient = getHttpClient();
    std::size_t tryNumber = 0;
    auto logger = makeLoggerWithRequestId(context->requestId());
    while (true) {
        try {
            const auto response = asyncRun(
                *httpClient,
                context,
                std::move(request),
                context->yieldContext()
            );
            if (response.status != 200) {
                auto msg = fmt::format("request to hound for mimes failed with status={}, reason={}", response.status, response.reason);
                auto error_code = makeHoundError(HoundError::httpError);
                LOGDOG_(logger, error, logdog::message=std::move(msg), logdog::error_code=std::move(error_code));
                if (!helpers::retriableCode(response.status) || ++tryNumber > config.retries) {
                    throw std::runtime_error("request for stid and mime parts failed with http error");
                }
                continue;
            }

            const auto contentType = response.headers.find("content-type");
            if (contentType == response.headers.end()) {
                auto msg = fmt::format("hound response has no content-type body={}", response.body);
                auto error_code = makeHoundError(HoundError::noContentType);
                LOGDOG_(logger, error, logdog::message=std::move(msg), logdog::error_code=std::move(error_code));
                throw std::runtime_error("no content type in response of request for stid and mime parts");
            }

            if (const auto& type = contentType->second; type != "application/json") {
                auto msg = fmt::format("hound response has unsupported content_type={}, body={}", type, response.body);
                auto error_code = makeHoundError(HoundError::invalidContentType);
                LOGDOG_(logger, error, logdog::message=std::move(msg), logdog::error_code=std::move(error_code));
                throw std::runtime_error("unsupported content type in response of request for stid and mime parts");
            }

            auto parsed = fromJson<HoundMimes>(response.body);
            if (parsed.error.is_initialized()) {
                const auto& parsedErr = parsed.error.get();
                std::string msg = "request to hound for mimes failed";
                auto error_code = mail_errors::error_code{parsedErr.code, houndErrorCategory(), parsedErr.reason};
                LOGDOG_(logger, error, logdog::message=std::move(msg), logdog::error_code=std::move(error_code));

                if (!isRetriableHoundError(HoundError(parsedErr.code)) || ++tryNumber > config.retries) {
                    throw std::runtime_error("request for stid and mime parts failed with server error");
                }
                continue;
            }

            if (!parsed.mimes.is_initialized()) {
                auto msg = fmt::format("hound response doesn't contain error or mimes body={}", response.body);
                auto error_code = makeHoundError(HoundError::invalidMimeBody);
                LOGDOG_(logger, error, logdog::message=std::move(msg), logdog::error_code=std::move(error_code));
                throw std::runtime_error("invalid response format of request for stid and mime parts");
            }

            auto messageParts = parsed.mimes->find(mid);
            return messageParts == parsed.mimes->end() ? boost::none : boost::make_optional(std::move(messageParts->second));

        } catch (const boost::system::system_error& e) {
            LOGDOG_(logger, error, logdog::exception=e);
            if (++tryNumber > config.retries) {
                throw std::runtime_error("http exception when request for messsage parts");
            }
        }
    }
}

} // namespace retriever
