#pragma once

#include <boost/system/error_code.hpp>

namespace retriever {

enum class HoundError {
    ok                          = 0,     /// No error
    unknown                     = 1,     /// Unknown error
    internal                    = 31,    /// internal error
    dbUnknownError              = 1000,  /// Unknown DB error
    dbReadOnly                  = 1001,  /// Database is in read-only mode
    dbUniqueConstraint          = 1003,  /// Unique constrain violated
    notInitialized              = 1006,  /// User is not initialized

    noSuchMessage               = 5000,  /// No such message
    invalidArgument             = 5001,  /// Invalid argument
    noSuchFolder                = 5002,  /// No such folder
    noSuchLabel                 = 5005,  /// No such label
    folderNotEmpty              = 5012,  /// Folder to be deleted is not empty
    revisionNotFound            = 5013,  /// Given revision can not be found
    tooManyMessages             = 5014,  /// User has too many messages
    noSuchTab                   = 5021,  /// No such tab
    emptyResult                 = 5022,

    uidNotFound                 = 7001,  /// Uid not found
    regInProgress               = 7002,  /// User registration in progress



    httpError          = -1,            /// Error during http request
    noContentType      = httpError-1,   /// Content type isn't present
    invalidContentType = httpError-2,   /// Invalid content type
    invalidMimeBody    = httpError-3,   /// Invalid mime body
};


struct HoundErrorCategory : boost::system::error_category {
public:
    const char* name() const noexcept override final {
        return "HoundErrorCategory";
    }

    std::string message(int value) const override {
        return "error code: " + std::to_string(value);
    }
};

inline boost::system::error_category& houndErrorCategory() {
    static HoundErrorCategory category;
    return category;
}

bool isRetriableHoundError(HoundError value);

}

namespace boost {
namespace system {

template<>
struct is_error_code_enum<retriever::HoundError> : public std::true_type {};

} // namespace system
} // namespace boost

