#include <src/services/resize/resize_client_impl.hpp>
#include <src/services/resize/resize_params_writer.hpp>
#include <src/services/resize/resize_errors.hpp>
#include <mail_errors/error_code.h>
#include <mail/http_getter/client/include/client.h>
#include <src/task_context.hpp>
#include <yplatform/encoding/url_encode.h>
#include <yamail/data/reflection/details/types.h>
#include <src/log.hpp>

namespace retriever {

using ErrorCode = mail_errors::error_code;

ResizeClientImpl::ResizeClientImpl(GetClusterClient getGenurlClusterClient, GetHttpClient getGetHttpClient, GetServiceTicket getServiceTicket, Config config)
    : getGenurlClusterClient(std::move(getGenurlClusterClient))
    , getGetHttpClient(std::move(getGetHttpClient))
    , getServiceTicket(std::move(getServiceTicket))
    , config(std::move(config)) {
}

std::string ResizeClientImpl::genurl(const TaskContextPtr& context, const GenurlParams& params) const {
    using yamail::data::reflection::applyVisitor;
    using yamail::data::reflection::namedItemTag;

    std::ostringstream requestUrl;
    requestUrl << "/genurl?";
    ResizeParamsWriter writer(requestUrl);
    applyVisitor(params, writer, namedItemTag(""));

    using namespace http_getter;
    Request request = http_getter::get(requestUrl.str())
        .headers(
            requestId=context->requestId(),
            serviceTicket=getServiceTicket("mulcagate", context->requestId())
        )
        .make();

    auto response = performWithClusterClient(std::move(request), context);
    if (!response) {
        throw std::runtime_error("resize genurl failed with no http response");
    }

    if (response->status != 200) {
        throw std::runtime_error("resize genurl failed with http error: status=" + std::to_string(response->status));
    }

    const auto logger = makeLoggerWithRequestId(context->requestId());
    const auto contentType = response->headers.find("content-type");

    if (contentType == response->headers.end()) {
        auto msg = fmt::format("Resize response has no content-type body={}", response->body);
        auto err_code = ErrorCode(ResizeError::noContentType);
        LOGDOG_(logger, error, logdog::message=std::move(msg), logdog::error_code=std::move(err_code));
        throw std::runtime_error("no content type in response for request to resize/genurl");
    }

    if (const auto& type = contentType->second; type != "text/plain") {
        auto msg = fmt::format("Resize response has unsupported content_type={}, body={}", type, response->body);
        auto err_code = ErrorCode(ResizeError::invalidContentType);
        LOGDOG_(logger, error, logdog::message=std::move(msg), logdog::error_code=std::move(err_code));
        throw std::runtime_error("unsupported content type in response for request to resize/genurl");
    }

    return response->body;
}

boost::optional<ResizeClient::Image> ResizeClientImpl::get(const TaskContextPtr& context, std::string url) const {
    using namespace http_getter;
    Request request = http_getter::get(std::move(url))
        .headers(
            requestId=context->requestId(),
            serviceTicket=getServiceTicket("mulcagate", context->requestId())
        )
        .timeouts(config.get.timeouts)
        .make();

    auto response = performWithHttpClient(std::move(request), context, config.get.retries);
    if (!response || response->status != 200) {
        return boost::none;
    }
    Image result;
    const auto contentDisposition = response->headers.find("content-disposition");
    if (contentDisposition != response->headers.end()) {
        result.contentDisposition = std::move(contentDisposition->second);
    }
    const auto contentType = response->headers.find("content-type");
    if (contentType != response->headers.end()) {
        result.contentType = std::move(contentType->second);
    }
    result.content = std::move(response->body);
    return result;
}

std::optional<ResizeClientImpl::Response> ResizeClientImpl::performWithClusterClient(http_getter::Request&& request,
        const TaskContextPtr& context) const {
    try {
        return asyncRun(
            *getGenurlClusterClient(),
            context,
            std::move(request),
            context->yieldContext()
        );
    } catch (const boost::system::system_error& e) {
        LOGDOG_(makeLoggerWithRequestId(context->requestId()), error, logdog::exception=e);
    }

    return std::nullopt;
}

std::optional<ResizeClientImpl::Response> ResizeClientImpl::performWithHttpClient(
        http_getter::Request&& request, const TaskContextPtr& context, std::size_t retries) const {
    auto httpClient = getGetHttpClient();
    auto logger = makeLoggerWithRequestId(context->requestId());
    for (std::size_t i = 0; i < retries + 1; ++i) {
        try {
            const auto response = asyncRun(
                *httpClient,
                context,
                std::move(request),
                context->yieldContext()
            );
            if (response.status != 200) {
                auto msg = fmt::format("request to resize failed with status={}, reason={}", response.status, response.reason);
                auto err_code = ErrorCode(ResizeError::httpError);
                LOGDOG_(logger, error, logdog::message=std::move(msg), logdog::error_code=std::move(err_code));
                if (http_getter::helpers::retriableCode(response.status)) {
                    continue;
                }
            }
            return response;
        } catch (const boost::system::system_error& e) {
            LOGDOG_(logger, error, logdog::exception=e);
        }
    }

    return std::nullopt;
}

} // namespace retriever
