#pragma once

#include <boost/system/error_code.hpp>

namespace retriever {

enum class ResizeError {
    httpError = 0,
    noContentType,
    invalidContentType
};

class ResizeErrorCategory : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "ResizeErrorCategory";
    }

    std::string message(int ev) const noexcept override {
        using Err = ResizeError;
        switch (Err(ev)) {
            case Err::httpError:
                return "http error";
            case Err::noContentType:
                return "no content-type";
            case Err::invalidContentType:
                return "invalid content-type";
        }
        return "resize error";
    }

    static const ResizeErrorCategory& instance() {
        static ResizeErrorCategory value;
        return value;
    }
};

inline boost::system::error_code make_error_code(ResizeError e) {
    return boost::system::error_code(static_cast<int>(e), ResizeErrorCategory::instance());
}

}

namespace boost::system {

template<>
struct is_error_code_enum<retriever::ResizeError> : public std::true_type {};

} // namespace boost::system

