from argparse import ArgumentParser
from collections import namedtuple
from mail.unistat.cpp.cython.meters import (
    AccessLogCount,
    AccessLogCountByFirstStatusDigit,
    AccessLogCountByPathAndFirstStatusDigit,
    AccessLogRequestTimeHist,
    AccessLogRequestTimeHistByPath,
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
    SupervisorLogRestartMeters,
)
from mail.unistat.cpp.cython.count_by_condition import (
    Counter,
    Condition,
    Signal,
    Filter,
    update_signal_names,
)
from mail.retriever.unistat.cpp.run import (
    run as run_unistat,
)
from mail.retriever.unistat.cpp.errors import (
    HoundError,
    ResizeError,
    MessagePartError,
)
import os
import yaml
import logging
import mail.unistat.cpp.cython.logs as logs
import re


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


SERVICE_ENDPOINTS = [
    '/message_part_real',
]


HTTP_HIST_BUCKETS = (0, 20, 100, 300, 1000)


RESIZE_ERROR_CATEGORY = 'ResizeErrorCategory'
RESIZE_ERROR_SIGNALS = [
    Signal(cond=Condition.Equals(ResizeError.httpError),            name='request_to_resize_failed_http'),
    Signal(cond=Condition.Equals(ResizeError.noContentType),        name='request_to_resize_failed_bad_response'),
    Signal(cond=Condition.Equals(ResizeError.invalidContentType),   name='request_to_resize_failed_bad_response'),

    Signal(cond=Condition.AnyStr(),                                 name='resize_unknown_error'),
]


HOUND_ERROR_CATEGORY = 'HoundErrorCategory'
HOUND_ERROR_SIGNALS = [
    Signal(cond=Condition.Equals(HoundError.httpError),             name='request_to_hound_failed_http'),
    Signal(cond=Condition.Equals(HoundError.noContentType),         name='request_to_hound_failed_bad_response'),
    Signal(cond=Condition.Equals(HoundError.invalidContentType),    name='request_to_hound_failed_bad_response'),
    Signal(cond=Condition.Equals(HoundError.invalidMimeBody),       name='request_to_hound_failed_bad_response'),

    Signal(cond=Condition.AnyStr(),                                 name='request_to_hound_failed_bad_request'),
]


MESSAGE_PART_ERROR_CATEGORY = 'MessagePartErrorCategory'
MESSAGE_PART_ERROR_SIGNALS = [
    Signal(cond=Condition.Equals(MessagePartError.invalidSid),          name='sid_invalid_data'),
    Signal(cond=Condition.Equals(MessagePartError.oldSid),              name='sid_old_format'),
    Signal(cond=Condition.Equals(MessagePartError.illFormedSid),        name='sid_unknown_format'),
    Signal(cond=Condition.Equals(MessagePartError.expiredSid),          name='sid_timed_out'),
    Signal(cond=Condition.Equals(MessagePartError.invalidAesKeyId),     name='sid_unknown_key'),
    Signal(cond=Condition.Equals(MessagePartError.invalidHmacKeyId),    name='sid_bad_integrity'),
    Signal(cond=Condition.Equals(MessagePartError.decryptionError),     name='sid_decryption_failure'),
    Signal(cond=Condition.Equals(MessagePartError.notFound),            name='no_message_part_for_uid_mid'),

    Signal(cond=Condition.AnyStr(),                                     name='message_part_unknown_error'),
]

COMMON_ERROR_SIGNALS = [
    Signal(cond=Condition.Contains('connection_timeout'),          name='connection_timeout'),
    Signal(cond=Condition.Contains('request_timeout'),             name='request_timeout'),
    Signal(cond=Condition.Contains('range message storage get'),   name='request_to_storage_failed'),
]


PART_ID_SIGNALS = [
    Signal(cond=Condition.Contains('part id is single message part'),   name='single_message_part'),
    Signal(cond=Condition.Contains('part id is multiple message part'), name='multiple_message_part'),
    Signal(cond=Condition.Contains('part id is temporary'),             name='temporary'),
    Signal(cond=Condition.Contains('part id is old'),                   name='old'),
]


ERROR_LEVEL_FILTER = Filter(key='level', cond=Condition.Equals('error'))


def create_filters(category):
    return [ERROR_LEVEL_FILTER, Filter(key='error_code.category', cond=Condition.Equals(category))]


def make_service_log_meters():
    meters = []
    prefix = 'retriever_error_{}'
    suffix = 'dmmm'
    error_code_key = 'error_code.value'

    signals = [
        (RESIZE_ERROR_SIGNALS, RESIZE_ERROR_CATEGORY),
        (HOUND_ERROR_SIGNALS, HOUND_ERROR_CATEGORY),
        (MESSAGE_PART_ERROR_SIGNALS, MESSAGE_PART_ERROR_CATEGORY),
    ]
    for sig, cat in signals:
        meters.append(Counter(error_code_key, update_signal_names(sig, prefix, suffix), create_filters(cat)))

    common_error_signals = update_signal_names(COMMON_ERROR_SIGNALS, prefix, suffix)
    meters.append(Counter('exception.what', common_error_signals, [ERROR_LEVEL_FILTER]))

    count_part_ids = update_signal_names(PART_ID_SIGNALS, 'part_ids_{}', 'summ')
    meters.append(Counter('message', count_part_ids, []))

    return meters


def make_access_log_meters(name_prefix, endpoints):
    meters = [
        AccessLogCount(name_prefix),
        AccessLogCountByFirstStatusDigit(name_prefix),
        AccessLogRequestTimeHist(HTTP_HIST_BUCKETS, 'access_log_request'),
    ]

    for endpoint in endpoints:
        meters += [
            AccessLogCountByPathAndFirstStatusDigit(endpoint, name_prefix),
            AccessLogRequestTimeHistByPath(HTTP_HIST_BUCKETS, endpoint, 'access_log_request'),
        ]

    return meters


def make_supervisor_meters():
    return [
        SupervisorLogRestartMeters('supervisor')
    ]


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='/app/log/unistat/unistat.log', help='path for yplatform.log')
    parser.add_argument('--supervisorlog', help='path for supervisord.log')
    parser.add_argument('service_config')
    return parser.parse_args()


def get_module_config(data, name):
    return next(v['configuration'] for v in data['config']['modules']['module'] if v['system']['name'] == name)


def get_logger_sink_path(data, name, sink_index=0):
    return os.path.join(os.curdir, data['config']['log'][name]['sinks'][sink_index]['path'])


def make_service_config(data):
    remove_scheme = re.compile(r"https?://")

    service_name = 'retriever'
    service_config = get_module_config(data, service_name)
    mulcagate_cc_config = get_module_config(data, service_config['mulcagate']['cluster_client_module'])
    resize_cc_config = get_module_config(data, service_config['resize']['genurl']['cluster_client_module'])

    storage_host = remove_scheme.sub('', mulcagate_cc_config['hosts']).strip()
    resize_genurl_host = remove_scheme.sub('', resize_cc_config['hosts']).strip()
    resize_get_host = remove_scheme.sub('',  service_config['resize']['get']['location']).strip()
    hound_host = remove_scheme.sub('',  service_config['hound']['location']).strip()
    return ServiceConfig(
        httpclient_log=get_logger_sink_path(data, 'http_client'),
        access_log=get_logger_sink_path(data, 'access'),
        service_log=get_logger_sink_path(data, service_name),

        storage_host=storage_host,
        resize_genurl_host=resize_genurl_host,
        resize_get_host=resize_get_host,
        hound_host=hound_host
    )


def make_http_client_log_meters(cfg):
    return [
        HttpClientHttpRequestCountByStatus(cfg.storage_host,       'count_by_status_storage'),
        HttpClientHttpRequestCountByStatus(cfg.hound_host,         'count_by_status_hound'),
        HttpClientHttpRequestCountByStatus(cfg.resize_genurl_host, 'count_by_status_resize_genurl'),
        HttpClientHttpRequestCountByStatus(cfg.resize_get_host,    'count_by_status_resize_get'),

        HttpClientHttpRequestTotalTimeHist(HTTP_HIST_BUCKETS, cfg.storage_host,         'time_hist_storage'),
        HttpClientHttpRequestTotalTimeHist(HTTP_HIST_BUCKETS, cfg.hound_host,           'time_hist_hound'),
        HttpClientHttpRequestTotalTimeHist(HTTP_HIST_BUCKETS, cfg.resize_genurl_host,   'time_hist_resize_genurl'),
        HttpClientHttpRequestTotalTimeHist(HTTP_HIST_BUCKETS, cfg.resize_get_host,      'time_hist_resize_get'),
    ]


ServiceConfig = namedtuple('ServiceConfig', (
    'httpclient_log',
    'access_log',
    'service_log',
    'storage_host',
    'resize_genurl_host',
    'resize_get_host',
    'hound_host'
))


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.service_config) as f:
        config = make_service_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    service_log = logs.TskvLog([], make_service_log_meters(), fast_forward, config.service_log)
    access_log = logs.AccessTskv([], make_access_log_meters('access_log', SERVICE_ENDPOINTS), fast_forward, config.access_log)
    http_client_log = logs.HttpClientLog([], make_http_client_log_meters(config), fast_forward, config.httpclient_log)

    logs_list = [service_log, access_log, http_client_log]

    if args.supervisorlog is not None:
        supervisor_log = logs.SupervisorLog(
            [],
            make_supervisor_meters(),
            fast_forward,
            args.supervisorlog
        )
        logs_list.append(supervisor_log)

    run_unistat(args.host, args.port, logs_list, args.log, logLevel='info')


if __name__ == '__main__':
    main()
