// Tracker API https://wiki.yandex-team.ru/tracker/api/#rabotastiketami

var tracker_api_url = "https://" + document.domain + "/st";
var arcanum_api_url = "https://" + document.domain + "/arcanum";

async function loadBoardCards(query) {
    if ($.isArray(query)) {
        let result = [];
        for (let subQuery of query) {
            result = result.concat(await loadBoardCards(subQuery));
        }
        return result;
    }

    var fields = [
        "assignee",
        "components",
        "createdAt",
        "createdBy",
        "deadline",
        "favorite",
        "followers",
        "id",
        "iterationCount",
        "key",
        "lastCommentUpdatedAt",
        "line",
        "remotelinks",
        "messageSource",
        "parent",
        "previousStatus",
        "priority",
        "project",
        "queue",
        "self",
        "sla",
        "status",
        "statusStartTime",
        "summary",
        "tags",
        "type",
        "updatedAt",
        "updatedBy",
        "version"
    ];
    var url = tracker_api_url + "/v2/issues?query=" + query + "&fields=" + fields.join(',') + "&perPage=800";
    return new Promise((resolve, reject) => {
        $.ajax({
            url: url,
            success: async function (answer) {
                resolve(parseCardList(answer));
            },
            error: function (xhr) {
                var error = String(xhr.status) + ' ' + xhr.statusText;
                console.log('loadBoardCards failed: ' + error);
                reject(error);
            }
        });
    });
}

async function moveIssue(issueKey, targetStatus) {
    var transition = await findTransition(issueKey, targetStatus);
    if (transition == "") return false;
    var issues_url = tracker_api_url + "/v2/issues";
    var url = issues_url + "/" + issueKey + "/transitions/" + transition + "/_execute";

    return new Promise((resolve, reject) => {
        $.ajax({
            url: url,
            method: "POST",
            success: async function (answer) {
                resolve(true);
            },
            error: function (xhr) {
                var error = String(xhr.status) + ' ' + xhr.statusText;
                console.log('moveIssue failed: ' + error);
                reject(error);
            }
        });
    });
}

async function loadUnreadPullRequests() {
    var url = arcanum_api_url + "/review/dashboard?view=incoming";
    return new Promise((resolve, reject) => {
        $.ajax({
            url: url,
            success: async function (answer) {
                var result = [];
                if (answer["reviews"]) {
                    answer["reviews"].forEach(item => {
                        if (item["unread"] === true && item["status"] === "pending_review") {
                            result.push(String(item["id"]));
                        }
                    });
                }
                resolve(result);
            },
            error: function (xhr) {
                var error = String(xhr.status) + ' ' + xhr.statusText;
                console.log('loadPullRequestState failed: ' + error);
                reject(error);
            }
        });
    });
}

function parseCardList(data) {
    var result = data.filter(function(item){
        return item["type"] != undefined && item["type"]["key"] != undefined;
    }).map(function(item){
        return parseCard(item);
    }).filter(function (card) { return card != undefined; });
    return fillParentInfo(result);
}

function parseCard(data) {
    var maskData = document.maskData;
    var result = {};
    result["id"] = data["key"];
    result["type"] = data["type"]["key"];
    result["queue"] = data["queue"]["key"];
    result["url"] = globalTrackerUrl + data["key"];
    if (data["assignee"]) {
        result["assignee"] = passOrMaskData(data["assignee"]["id"], maskData);
    }
    result["summary"] = passOrMaskData(data["summary"], maskData);
    result["components"] = [];
    $.each(data["components"], function (i, component) {
        result["components"].push(passOrMaskData(component["display"], maskData));
    });
    result["status"] = data["status"]["key"];
    result["statusStartTime"] = data["statusStartTime"];
    result["deadline"] = data["deadline"];
    result["priority"] = result["deadline"] ? "deadline" : data["priority"]["key"];
    result["inStatusTime"] = (result["status"] == "new"
        ? 0
        : dateDiff(new Date(data["statusStartTime"].slice(0,10)), new Date()));
    if (data["parent"]) {
        result["parent"] = {"id" : data["parent"]["key"]};
    }
    if (data["project"]) {
        result["project"] = {"id" : data["project"]["id"], "name": data["project"]["display"]};
    }
    result.tags = data["tags"] ? data["tags"] : [];
    for (let tag of result.tags) {
        if (tagIsBlocker(tag)) {
            result.blockingReason = blockerNameFromTag(tag);
            break;
        }
    }
    return result;
}

function fillParentInfo(cards) {
    $.each(cards, (_, cardToFillParent) => {
        if (!cardToFillParent["parent"]) return;
        $.each(cards, (_, card) => {
            if (card["id"] === cardToFillParent["parent"]["id"]) {
                cardToFillParent["parent"] = card;
                return false;
            }
        });
    });
    return cards;
}

async function loadPullRequests(issueKey) {
    var url = tracker_api_url + "/v2/issues/" + issueKey + "/remotelinks";
    return new Promise((resolve, reject) => {
        $.ajax({
            url: url,
            success: function (answer) {
                var pullRequests = [];
                $.each(answer, function (i, item) {
                    var appID = item["object"]["application"]["id"];
                    if (appID == "ru.yandex.arcanum") {
                        pullRequests.push(item["object"]["key"]);
                    }
                });
                resolve(pullRequests);
            },
            error: function (xhr) {
                var error = String(xhr.status) + ' ' + xhr.statusText;
                console.log('PR load for ' + issueKey + ' failed: ' + error);
                reject(error);
            }
        });
    });
}

async function loadPullRequestState(id) {
    var url = arcanum_api_url + "/v1/review-requests/" + id + "?fields=state";
    return new Promise((resolve, reject) => {
        $.ajax({
            url: url,
            success: async function (answer) {
                if (!answer['data'] || !answer['data']['state']) {
                    reject("invalid arcanum API response");
                    return;
                }
                resolve(answer['data']['state']);
            },
            error: function (xhr) {
                var error = String(xhr.status) + ' ' + xhr.statusText;
                console.log('loadPullRequestState failed: ' + error);
                reject(error);
            }
        });
    });
}

async function loadSubtasks(issueKey) {
    var url = tracker_api_url + "/v2/issues/" + issueKey + "/links";
    return new Promise((resolve, reject) => {
        $.ajax({
            url: url,
            success: function (answer) {
                var subtasks = [];
                $.each(answer, function (i, item) {
                    if (item["direction"] == "outward") {
                        subtasks.push(parseSubtaskCard(item));
                    }
                });
                subtasks = subtasks.sort((a, b) => {
                    var x = subtaskStatusToInt(a["status"]);
                    var y = subtaskStatusToInt(b["status"]);
                    return ((x > y) ? -1 : ((x < y) ? 1 : 0));
                });
                resolve(subtasks);
            },
            error: function (xhr) {
                var error = String(xhr.status) + ' ' + xhr.statusText;
                console.log('subtasks load for ' + issueKey + ' failed: ' + error);
                reject(error);
            }
        });
    });
}

async function findTransition(issueKey, targetStatus) {
    var issues_url = tracker_api_url + "/v2/issues";
    var url = issues_url + "/" + issueKey + "/transitions";
    return new Promise((resolve, reject) => {
        $.ajax({
            url: url,
            success: function (answer) {
                var result = parseAndFindTransition(answer, targetStatus);
                resolve(result);
            },
            error: function (xhr) {
                var error = String(xhr.status) + ' ' + xhr.statusText;
                console.log('findTransition failed: ' + error);
                reject(error);
            }
        });
    });
}

async function addTags(issueKey, tags) {
    var issues_url = tracker_api_url + "/v2/issues";
    var url = issues_url + "/" + issueKey;
    return new Promise((resolve, reject) => {
        $.ajax({
            url: url,
            method: "PATCH",
            data: JSON.stringify({"tags": {"add": tags}}),
            contentType: "application/json;charset=utf-8",
            success: async function (_) {
                resolve(true);
            },
            error: function (xhr) {
                var error = String(xhr.status) + ' ' + xhr.statusText;
                console.log('addTags failed: ' + error);
                reject(error);
            }
        });
    });
}

async function removeTags(issueKey, tags) {
    var issues_url = tracker_api_url + "/v2/issues";
    var url = issues_url + "/" + issueKey;
    return new Promise((resolve, reject) => {
        $.ajax({
            url: url,
            method: "PATCH",
            data: JSON.stringify({"tags": {"remove": tags}}),
            contentType: "application/json;charset=utf-8",
            success: async function (_) {
                resolve(true);
            },
            error: function (xhr) {
                var error = String(xhr.status) + ' ' + xhr.statusText;
                console.log('removeTags failed: ' + error);
                reject(error);
            }
        });
    });
}

function parseAndFindTransition(data, keyTo) {
    var result = "";
    $.each(data, (_, transition) => {
        if (keyTo == transition["to"]["key"]) {
            result = transition["id"];
            return false;
        }
    });
    return result;
}

function parseSubtaskCard(data) {
    var subtask = {};
    subtask["id"] = data["object"]["key"];
    subtask["status"] = data["status"]["key"];
    return subtask;
}

function isArcanumLink(item) {
    return item['display'].search('ru.yandex.arcanum') != -1;
}

function pullRequestFromLink(item) {
    return item['display'].match(/relates ([0-9]+)/)[1];
}

function dateDiff(date1, date2) {
    var msec = date2 - date1;
    var mins = Math.floor(msec / 60000);
    var hrs = Math.floor(mins / 60);
    var days = Math.floor(hrs / 24);
    return days;
}

function subtaskStatusToInt(status)
{
    if (status == "backlog") return 2;
    if (status == "inProgress") return 3;
    if (status == "inReview") return 4;
    if (status == "awaitingRelease") return 5;
    if (status == "open") return 900;
    if (status == "closed") return 999;
    return 99;
}

function passOrMaskData(data, maskData) {
    if (maskData) {
        return generateFakeData(data.length);
    }
    return data;
}

function generateFakeData(length) {
    var result           = [];
    var characters       = 'abcdefghijklmnopqrstuvwxyz0123456789      ';
    var charactersLength = characters.length;
    for ( var i = 0; i < length; i++ ) {
      result.push(characters.charAt(Math.floor(Math.random() * charactersLength)));
   }
   return result.join('');
}

function blockerPrefix() {
    return "blocked:";
}

function tagIsBlocker(tag) {
    return tag.includes(blockerPrefix());
}

function blockerNameFromTag(tag) {
    return tag.substring(blockerPrefix().length);
}

function tagFromBlockerName(reason) {
    return blockerPrefix() + reason;
}

async function _sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}