function drawHeader(label) {
    document.title = label;
}

function drawBoard(grid) {
    let bodyElement = $('#board-body');
    bodyElement.html('');
    $.each(grid, function (gridRowIndex, gridRow) {
        if (isEmptyRow(gridRow)) {
            return;
        }
        let rowElement = generateRowElement(templates, gridRow, gridRowIndex);
        bodyElement.append(rowElement);
    });

    alignScrollableColumns();
}

function drawStats(url) {
    if (url == "") return;
    let statsElement = $('#board-stats');
    statsElement.html('');
    let content = $(templates.stats);
    content.find('iframe').attr('src', url);
    statsElement.append(content);
}

function generateRowElement(templates, gridRow, gridRowIndex) {
    let rowElement = $(templates.row);
    $.each(gridRow.cols, function (colIndex, col) {
        let colsHTML = generateCellElement(templates, gridRow, gridRowIndex, col, colIndex);
        rowElement.append(colsHTML);
    });
    return rowElement;
}

function alignScrollableColumns() {
    $('#board-body').children('.row').each((irow,row)=>{
        let cols = $(row).children('.col');
        let maxHeight = 0;
        let scrollable = [];
        for (let i = 0; i < cols.length; i++) {
            if ($(cols[i]).hasClass('scrollable')) {
                scrollable.push($(cols[i]));
            } else {
                let height = $(cols[i]).children('.row').height();
                maxHeight = Math.max(maxHeight, height);
            }
        }
        if (maxHeight === 0) maxHeight = 400;
        maxHeight += 140;
        for (let i = 0; i < scrollable.length; i++) {
            $(scrollable[i]).children('.row').css('max-height', maxHeight);
        }
    });
}

function generateCellElement(templates, row, rowIndex, col, colIndex) {
    // TODO don't re-calculate this values for ever column
    var realColCount = calcRealColCount(row);
    var hasLargeCols = realColCount > row.cols.length;

    let result = $(templates.column);
    // ID
    let colID = "cell_" + rowIndex + "_" + colIndex;
    result.attr("id", colID);
    result.data('status', col.status);
    // class
    let colClass = 'col';
    if (hasLargeCols) {
        if (col.large) {
            colClass = "col large";
        } else {
            colClass = 'col-' + Math.floor(12 / realColCount);
        }
    }
    result.addClass(colClass);
    result.addClass('col-status');

    let cards = col.cards;
    if (!cards) cards = [];
    let adjacentCards = col.adjacentCards ? col.adjacentCards : [];
    let totalLength = cards.length + (adjacentCards ? adjacentCards.length : 0);
    let counterLabel = col.title ? totalLength : "";
    if (col.scrollable) {
        result.addClass('scrollable');
    }
    let counterStyle = "text-dark";
    if (col.limit > 0) counterLabel = totalLength + '/' + col.limit;
    if (col.limit > 0 && totalLength > col.limit) counterStyle = "text-danger";
    // label
    result.find('span.col-label').html(col.title);
    result.find('span.col-counter').html(counterLabel);
    result.find('span.col-counter').addClass(counterStyle);
    result.on('dragover', function (event) { allowDrop(event); }); // XXX
    result.on('drop', function (event) { drop(event); }); // XXX

    $.each(cards, function (_, card) {
        let cardElement = generateCardElement(templates.card, row, rowIndex, col, colIndex, card);
        result.find('.cards').append(cardElement);
    });

    if (col.adjacentTitle) {
        let adjacentCell = $(templates.column);
        let colID = "cell_" + rowIndex + "_" + colIndex + "_adjacent";
        adjacentCell.attr("id", colID);
        adjacentCell.data('status', col.adjacentStatus);
        adjacentCell.find('span.col-label').html(col.adjacentTitle);
        adjacentCell.addClass('adjacent');
        adjacentCell.addClass('col-status');
        $.each(adjacentCards, function (_, card) {
            let cardElement = generateCardElement(templates.card, row, rowIndex, col, colIndex, card);
            adjacentCell.find('.cards').append(cardElement);
        });
        result.append(adjacentCell);
    }

    return result;
}

function generateCardElement(template, row, rowIndex, col, colIndex, card) {
    let result = $(template);
    result.attr('id', 'card_' + card.id);
    result.data('cardid', card.id);
    result.data('cardurl', card.url);
    result.find('.card-link').html(card.id);
    result.find('.card-link').attr('href', card.url);
    result.find('.card-link').data('cardurl', card.url);
    result.addClass(colorFor(card));

    if (card.url.length > 0) { // XXX draggable
        result.attr('draggable', true);
        result.on('dragstart', function (event) { drag(event); }); // XXX
        result.dblclick(function (event) { onCardOpen(event); });
    }

    result.find('.fa-copy').click(function (event) { onCopyIDClick(event); });
    result.find('.fa-bars').click(function (event){ openCardMenu(event); });
    result.find('.card-link-line').dblclick(function (event){ event.preventDefault(); return false; });

    result.find('.card-menu-lock').click(function (event) { onBlockerAddClick(event); });
    result.find('.card-menu-unlock').click(function (event) { removeBlockers(event); });
    result.find('.card-text').html(card.summary);

    result.find('.card-assignee').append(generateAssigneeHTML(card));
    result.find('.card-assignee').addClass('card-' + card.assignee);

    result.find('.card-markers').append(markersFor(card));
    result.find('.card-components').append(componentsToString(card));

    if (card.deadline) {
        result.find('.card-deadline').html('🗓️ ' + card.deadline);
        result.find('.card-deadline').removeClass('hidden');
    }

    if (card.blockingReason) {
        result.find('.card-blocker').append(card.blockingReason.replaceAll("_"," "));
        result.find('.card-blocker').removeClass('hidden');
        result.find('.card-menu-unlock').removeClass('hidden');
        result.find('.card-menu-unlock').click(function (event) { removeBlockers(event); });
    } else {
        result.find('.card-menu-lock').removeClass('hidden');
    }

    result.find('.card-age').append(durationFor(card));
    result.find('.card-pull-requests').attr('id', 'card_'+card.id+'_pull_requests');
    result.find('.card-children').attr('id', 'card_' + card.id + '_children');
    return result;
}

function colorFor(card) {
    if (card.priority == "critical" || card.priority == "blocker") {
        return "card-color-alert";
    }
    if (card.priority == "minor" || card.priority == "trivial") {
        return "card-color-minor";
    }
    if (card.priority == "deadline") {
        return "card-color-deadline";
    }
    // if (card["type"] == "complaint" || card["type"] == "incident") {
    //     cardColor = "card-color-service";
    // }
    // if (outdated(card)) {
    //     cardColor = "card-color-alert";
    // }
    if (card.id == "") {
        return "card-color-empty";
    }
    return "card-color-default";
}

function markersFor(card) {
    var markers = "";
    if (card.type == "bug") markers += "🐞";
    // if (card.priority == "critical") markers += " 🚩";
    if (card.type == "complaint") markers += "😥";
    if (card.type == "service") markers += "⚒️";
    if (card.type == "project") markers += "💼";
    if (card.type == "research") markers += "🔬";
    if (card.type == "incident") markers += "🔥";
    return markers + " "; // XXX
}

function durationFor(card) {
    return durationStr = card.inStatusTime > 0 ? card.inStatusTime + "d" : "";
}

function componentsToString(card) {
    return card.components ? card.components.join() : "";
}

function generateAssigneeHTML(card) {
    var result = "";
    if (card.assignee) {
        var firstLetter = card.assignee[0];
        var result = card.assignee.slice(1);
        const makeAvatarUrl = (login) => `https://center.yandex-team.ru/api/v1/user/${login}/avatar/60.jpg`
        result = '<span class="text">'
            + '<img class="avatar" data-login="' + card.assignee + '" src="' + makeAvatarUrl(card.assignee) + '">'
            + '</span>';
    }
    return result;
}

function drawPullRequestForCard(prID, cardID, state) {
    var elementID = "#card_" + cardID + "_pull_requests";
    var element = $(elementID);
    var html = element.html();
    var prUrl = globalPullRequestUrl + prID;
    var stateIcon = "?";
    if (state == "unread") {
        stateIcon = "⌛";
    } else if (state == "closed") {
        stateIcon = "👍";
    } else {
        stateIcon = "🥕";
    }
    html += '<span><a class="pr" href="' + prUrl + '">' + stateIcon + '</a></span>';
    element.html(html);
}

function drawChildrenForCard(cardID, total, closed) {
    var elementID = "#card_" + cardID + "_children";
    var element = $(elementID);
    element.html(templates.childrenScore);
    element.find('.bar-container').css('width', Math.round(100 * closed / total) + '%');
}

function allowDrop(ev) {
    ev.preventDefault();
}

function drag(ev) {
    let cardID = $(ev.target).data("cardid");
    ev.originalEvent.dataTransfer.setData("cardid", cardID);
    console.log(ev.originalEvent.dataTransfer.getData("cardid"));
}

function drop(ev) {
    ev.preventDefault();
    var cardID = ev.originalEvent.dataTransfer.getData("cardid");
    var targetStatus = $(ev.target).closest(".col-status").data("status");
    moveCard(cardID, targetStatus);
}

function onCopyIDClick(ev) {
    let target = $(ev.target);
    let cardID = target.closest(".card").data("cardid");
    directCopy(cardID);
};

function onCardOpen(ev) {
    ev.preventDefault();
    let target = $(ev.target);
    let url = target.closest(".card").data("cardurl");
    openCardFrame(url);
};

function openCardFrame(url) {
    $('#card-frame-container').css('top', $(window).scrollTop());
    $('#card-frame').attr('src', url);
    $('#card-frame-container').show();
};

function closeCardFrame() {
    $('#card-frame').attr('src', 'about:blank');
    $('#card-frame-container').hide();
}

function openCardMenu(event) {
    let card = $(event.target).closest(".card")
    card.find('.dropdown-toggle').dropdown();
}

function onBlockerAddClick(ev) {
    let target = $(ev.target);
    let cardID = target.closest(".card").data("cardid");
    openBlockingDialog(cardID);
    ev.preventDefault();
};

function openBlockingDialog(cardID) {
    let dialog = $(dialogs.test);
    dialog.dialog({
        autoOpen: false,
        width: 400,
        modal: true,
        buttons: [
        {
            text: "OK",
            click: function() {
                var reason = $('input[name="reason"]').val();
                reason = reason.replaceAll(" ", "_");
                if (reason && reason != "") {
                    blockCard(cardID, reason);
                    $( this ).dialog( "close" );
                }
            }
        },
        ]
    });
    dialog.dialog("open");
}

function removeBlockers(event) {
    let target = $(event.target);
    let cardID = target.closest(".card").data("cardid");
    removeBlockersFromCard(cardID);
}

function outdated(card) {
    return !noTimedAlertStatuses.includes(card["status"])
        && !noTimedAlertTypes.includes(card["type"])
        && card["inStatusTime"] >= maxDaysTicketInState;
}

function isEmptyRow(row) {
    for (let i = 0; i < row.cols.length; i++) {
        if (row.cols[i].cards.length > 0) return false;
        if (row.cols[i].adjacentCards && row.cols[i].adjacentCards.length > 0) return false;
    }
    return true;
}

function directCopy(str) {
    //based on https://stackoverflow.com/a/12693636
    var tmp = document.oncopy
    document.oncopy = function (event) {
        event.clipboardData.setData("Text", str);
        event.preventDefault();
    };
    document.execCommand("Copy");
    document.oncopy = tmp;
}

function calcRealColCount(row) {
    var realColCount = 0;
    for (let i = 0; i < row.cols.length; i++) {
        realColCount += (row.cols[i].large ? 2: 1);
    }
    return realColCount;
}
