#!/usr/bin/python3
# coding: utf-8

from pathlib import Path
from logging.handlers import RotatingFileHandler
from http.server import HTTPServer, SimpleHTTPRequestHandler
import logging
import socket
import json
import time
import os

port = 11010
keywords_to_ignore = ['PASSIVE-CHECK']

if 'QLOUD_DATACENTER' in os.environ:
    geo = os.environ.get('QLOUD_DATACENTER')
else:
    geo = os.environ.get('DEPLOY_NODE_DC')

common_prefix = 'geo=' + geo + ';' if geo is not None else ''

log_size = 1024 * 1024 * 4

def setup_logging():
    log_path = Path('/var/log/unistat/unistat.log')
    if not log_path.parent.exists():
        log_path.parent.mkdir()
    log_path.touch(exist_ok=True)
    loghandler = RotatingFileHandler(str(log_path), maxBytes=log_size, backupCount=1)
    
    LOGFORMAT = 'timestamp=%(asctime)s\t%(message)s'

    logging.basicConfig(format=LOGFORMAT,
                        level=logging.INFO,
                        handlers=(loghandler,))
    logging.getLogger("urllib3").setLevel(logging.WARNING)

def main():
    setup_logging()
    server = HTTPServerV6(('::', port), UnistatHandler)
    server.serve_forever()


class HTTPServerV6(HTTPServer):
    address_family = socket.AF_INET6


class UnistatHandler(SimpleHTTPRequestHandler):
    def log_message(self, format, *args):
        logging.info('ip=%s\tmethod=%s\trequest=%s\tcode=%s\texectime=%s',
            self.client_address[0],
            self.command,
            self.path,
            self.code,
            '%.6f' % (time.time() - self.start_exec_time))

    def log_error(self, format, *args):
        if hasattr(self, 'error_msg'):
            logging.info('ip=%s\trequest=%s\terror=%s',
                self.client_address[0],
                self.path,
                self.error_msg)

    def send_response(self, code, message=None):
        self.code = code
        SimpleHTTPRequestHandler.send_response(self, code, message)

    def send_error(self, code, message=None, explain=None):
        self.code = code
        SimpleHTTPRequestHandler.send_error(self, code, message, explain)

    def do_GET(self):
        self.start_exec_time = time.time()
        if self.path == '/unistat':
            try:
                response_body = self.unistat_response_body()
                self.send_response(200)
                self.send_header('Content-type', 'text/json')
                self.end_headers()
                self.wfile.write(bytes(response_body, 'UTF-8'))
            except BaseException as e:
                self.error_msg = str(e) if len(str(e)) else str(type(e))
                self.send_error(500, 'Failed to get unistat metrics')
        elif self.path == '/ping':
            self.send_response(200)
            self.end_headers()
            self.wfile.write(b'pong')
        else:
            self.send_error(404)

    def unistat_response_body(self):
        with open('/var/log/metrics/metrics', 'r') as metrics_file:
            metrics = metrics_file.readlines()

        yasm_metrics = []
        for metric in metrics:
            yasm_metrics += self.to_yasm_format(metric)

        return json.dumps(yasm_metrics)

    def to_yasm_format(self, metric):
        for keyword in keywords_to_ignore:
            if metric.find(keyword) != -1:
                return []

        metric_parts = metric.strip().split(' ')
        if len(metric_parts) == 3:
            name, value, _ = metric_parts
            tag = 'none'
        elif len(metric_parts) == 4:
            name, value, _, tag = metric_parts
            if not tag:
                tag = 'none'
        else:
            return []

        result = []
        name = name.replace('.', '_')
        value = json.loads(value)
        if name.endswith('_cumulative_hgram'):
            name = name[:-len('_cumulative_hgram')]
            result.append([self.format_metric_name(tag, name, '_dhhh'), value])
        elif name.endswith('_hgram'):
            name = name[:-len('_hgram')]
            result.append([self.format_metric_name(tag, name, '_hhhh'), value])
        elif name.endswith('_max'):
            name = name[:-len('_max')]
            result.append([self.format_metric_name(tag, name, '_axxx'), value])
        elif 'timings' in name or 'histogram' in name:
            result.append([self.format_metric_name(tag, name, '_hhhh'), value])
        elif name.endswith('_cumulative'):
            name = name[:-len('_cumulative')]
            result.append([self.format_metric_name(tag, name, '_dhhh'), value])
            result.append([self.format_metric_name(tag, name, '_deee'), value])
        else:
            result.append([self.format_metric_name(tag, name, '_hhhh'), value])
            result.append([self.format_metric_name(tag, name, '_aeee'), value])
        return result

    def format_metric_name(self, tag, name, suffix):
        return '{}ctype={};{}{}'.format(common_prefix, tag, name, suffix)

if __name__ == '__main__':
    main()
