#! /bin/bash

[[ "$TRACE" ]] && set -x
set -eo pipefail

if uname | grep -q Darwin; then
    which greadlink &>/dev/null || { echo "Error: greadlink not found, please install coreutils" 1>&2; exit 1; }
    readlink="greadlink"
fi
[[ "$readlink" ]] || readlink="readlink"

me="$("$readlink" -f "$0")"
setup_root="$(dirname "$me")"
format_file="$setup_root/.clang-format"
dirs_to_format="$setup_root/dirs_to_format"

function find_mail_root()
{
    my_path="$setup_root"
    while [[ "$my_path" != "." ]]; do
        my_path="$(dirname "$my_path")"
        if [[ "$my_path" == */mail ]]; then
            echo "$my_path"
            exit 0;
        fi
    done;
    exit 1;
}

mail_root="$(find_mail_root)"

function format_dir
{
    dir="$1"
    format_file="$2"

    cp "$format_file" "$dir/.clang-format"
    find "$dir" \( -name '*.h' -or -name '*.cc' -or -name '*.hpp' -or -name '*.cpp' -or -name '*.ipp' \) -execdir clang-format -i -fallback-style=none {} \;
    rm "$dir/.clang-format"
}

function format_all
{
    for dir in $(<"$dirs_to_format"); do
        dir="$mail_root/$dir"
        format_dir "$dir" "$format_file"
    done
}

function extension-hook
{
    args=("$@")
    # To apply our custom format, replace -assume-filename
    # with a fake pointing to the directory containing our custom format.
    for ((i=1; i<"$#"; ++i)); do
        if [[ "${args[i]}" =~ -assume-filename=./* ]]; then
            assume_filename_arg="${args[i]}"
            args["$i"]="-assume-filename=$setup_root/fake.cc"
        fi
    done;
    [[ "$assume_filename_arg" ]] || exit 1;

    # Check whether we need to format the file.
    for dir in $(<"$dirs_to_format"); do
        [[ "$assume_filename_arg" =~ "$mail_root/$dir".* ]] && need_format=yes
    done
    [[ "$need_format" ]] || exit 0;

    clang-format "${args[@]}"
}

case "$1" in
help | -help | --help | -h )
    echo 'usage: apply_clang_format all|<extension-args>'
    ;;
all )
    format_all
    exit "$?"
    ;;
* )
    extension-hook "$@"
    exit "$?"
    ;;
esac
