#!/bin/bash

#default values
logrotate_wait=90

usage() {
    echo "usage: $0 <deploy|qloud> --app app --env env --componet component [--qloud-group group] [--logrotate-wait seconds]"
    echo "examples:"
    echo "    connect to d:mail_pop3_testing.main"
    echo "       $0 deploy --app pop3 --env testing --component main"
    echo "    connect to mail_pop3_testing_pop3"
    echo "       $0 qloud --app pop3 --env testing --qloud-group mail_pop3_testing_pop3"
    echo "       $0 qloud --app pop3 --env testing --component pop3"
}

check_host_arg() {
    if ! [ $host = "deploy" ] && ! [ $host = "qloud" ]; then
        usage
        exit 1
    fi
}

check_required_args() {
    if [[ -z "$host" ]] || [[ -z "$app" ]] || [[ -z "$env" ]] || ( [ $host = "deploy" ] && [[ -z "$component" ]] ); then
        usage
        exit 1
    fi
}

check_args() {
    check_host_arg
    check_required_args
}

if [[ -z "$1" ]]; then
    usage
    exit 0
fi

host="$1"
shift

while [[ -n "$1" ]]; do
    case "$1" in
        -h | --help)
            usage
            exit 0
            ;;
        --app)
            shift
            app=$1
            ;;
        --env)
            shift
            env=$1
            ;;
        --component)
            shift
            component=$1
            ;;
        --qloud-group)
            shift
            qloud_group=$1
            ;;
        --logrotate-wait)
            shift
            logrotate_wait=$1
            ;;
        *)
            usage
            exit 1
            ;;
    esac
    shift
done

check_args

get_address() {
    if [ $host = "deploy" ]; then
        local address=$(executer --cached --user root --quiet -c hostlist d:mail_${app}_${env}.${component} 2>&1 | grep "^app." | head -n1)
        address=${address:4}
    else
        if [[ -z "$qloud_group"  ]]; then
            qloud_group="mail_${app}_${env}_${component}"
        fi
        local address=$(executer --quiet -c hostlist %$qloud_group 2>&1 | grep qloud | head -n1)
    fi

    echo $address
}

execute() {
    local address="$1"
    local command="$2"
    local container="$3"

    if [ $host = "deploy" ]; then
        executer --cached --quiet --user root -c "exec $container.$address $command"
    else
        executer --cached --quiet -c "exec $address $command"
    fi
}

check_push_client() {
    local address="$1"

    echo "[push-client]"

    local request="push-client --status -c /app/push-client/$env.yml 2>&1"
    local container="push-client"
    local timestamp_prefix="^[0-9]{2}\.[0-9]{2}\.[0-9]{4}-[0-9]{2}\.[0-9]{2}\.[0-9]{2}:"

    local status=$(execute "$address" "$request" "$container")

    if (!(echo "$status" | grep -qE "${timestamp_prefix} status: ok$") \
        || !(echo "$status" | grep -qE "${timestamp_prefix} *auth status: ok$")); then
        echo "Not ok"
        echo "$status"
        return 1
    fi

    echo "Ok"
}

check_unistat() {
    local address="$1"

    echo "[unistat]"

    local container="unistat"

    local request="curl -i 'localhost:11010/unistat' 2>/dev/null"
    if !( execute "$address" "$request" "$container" | grep -qe "HTTP.*200 OK"); then
        echo "Got error code"
        return 1
    fi

    local request="curl 'localhost:11010/unistat' 2>/dev/null"
    if !( execute "$address" "$request" "$container" | grep -qe "reactor"); then
        echo "No reactor fields"
        return 1
    fi

    echo "Ok"
}

check_logrotate() {
    local address="$1"

    echo "[logrotate]"

    local container="cron"

    local cur_time=$(date +'%H:%M')

    local request="sed -i 's/0 0 \* \* \*/\* \* \* \* \*/g' /etc/cron.d/logrotate-daily"
    execute "$address" "$request" "$container"

    echo "Waiting $logrotate_wait seconds"
    sleep $logrotate_wait

    local request="sed -i 's/\* \* \* \* \*/0 0 \* \* \*/g' /etc/cron.d/logrotate-daily"
    execute "$address" "$request" "$container"

    local request="ls -l /app/log/"
    local updated=false
    local update_times=$(execute "$address" "$request" "$container" | grep "log.0" | awk '{print $8}')
    for time in $update_times; do
        if [[ "$time" > "$cur_time" ]] || [[ "$time" == "$cur_time" ]]; then
            updated=true
            break
        fi
    done

    if [ "$updated" != true ]; then
        echo "Logs not updated"
        echo "Start time: $cur_time"
        execute "$address" "$request" "$container" | grep "log.0"
        return 1
    fi

    echo "Ok"
}

check_app() {
    local address="$1"

    echo "[app]"

    local container="app"

    local request="ls -la /proc/\$(pgrep app)/fd"
    if (execute "$address" "$request" "$container" | grep -q "deleted"); then
        echo "Using deleted logs"
        return 1
    fi

    local request="ls -la /proc/\$(pgrep app)/fd"
    if (execute "$address" "$request" "$container" | grep -q "log.0"); then
        echo "Using old logs"
        return 1
    fi

    echo "Ok"
}

address=$(get_address)

echo "Checking $address"

check_push_client "$address"
check_unistat "$address"
check_logrotate "$address"
check_app "$address"
