#!/usr/bin/env bash

[[ "$TRACE" ]] && set -eux

app=${1?need app}
path=${2?need path}

path=$(realpath $path)
arcroot=$(ya dump root)

info () {
    local msg=$1
    echo "INFO   : $msg"
}

warning () {
    local msg=$1
    echo "WARNING: $msg"
}

pushd () {
    command pushd "$@" > /dev/null
}

popd () {
    command popd "$@" > /dev/null
}

pushd $path

# var/log/${oldname}/${oldname}.log -> app/log/app.log
# var/log/${oldname}/ -> app/log/
find . -type f | xargs sed -r -i "s/var\/log\/($app|\\\$QLOUD_APPLICATION)/app\/log/g"
find . -type f | xargs sed -r -i "s/($app|\\\$QLOUD_APPLICATION)\.log/app\.log/g"

# tvm_secret_file: etc/${oldname}/tvm_secret -> secret_file: app/secrets/tvm_secret
find . -type f | xargs sed -r -i "s/etc\/$app\/tvm_secret/app\/secrets\/tvm_secret/g"

# cert_file: etc/pop3/ssl -> cert_file: app/secrets
# key_file: etc/pop3/ssl -> cert_file: app/secrets
find . -type f | xargs sed -r -i "s/etc\/$app\/ssl/app\/secrets/g"


pushd ${arcroot}/mail/service-management/ci

if [[ $app == "collectors-ext" ]]; then
    ci_app="collectors_ext"
else
    ci_app=$app
fi

envs=$(./ci ls-apps 2>/dev/null | grep ${ci_app} | awk '{print $2}')
for env in $envs; do
    ./ci dump $ci_app $env 2>/dev/null | fgrep '"target"' | awk '{match($0,/: "(.*)"/,m); print m[1]}'
done | sort | uniq > /tmp/tmp.qloud_secrets

popd


for secret in $(grep -v -e "/etc/xivamob/tvm_secret" -e "/etc/xivamob/ssl/" -e "root.crt" /tmp/tmp.qloud_secrets); do
    warning "trying to process specific secret file: ${secret}; check it manually"
    new_secret_path="app/secrets/$(basename ${secret})"
    secret=${secret:1}

    find . -type f | xargs sed -r -i "s/${secret//\//\\/}/${new_secret_path//\//\\/}/g"
done

# /etc/${oldname}/ -> /app/config
find . -type f | xargs sed -r -i "s/etc\/$app/app\/config/g"

# uid: ${oldname} -> uid: app
# gid: ${oldname} -> gid: app
find ./etc -type f | xargs sed -r -i "s/uid: (xiva|$app)/uid: app/g"
find ./etc -type f | xargs sed -r -i "s/gid: (xiva|$app)/gid: app/g"


# mail-trusty-common -> mail-base-focal
# TODO: resolve the situation when the version is not the latest
find . -type f | xargs sed -i "s/mail-trusty-common/mail-base-focal/g"

if (grep -qr "search-trusty" .); then
    warning "found \"search-trusty\"; perhaps already included in \
        https://a.yandex-team.ru/arcadia/mail/docker-images/mail-base/deploy/etc/apt/sources.list.d/yandex.list"
fi

# /usr/sbin/entrypoint.sh -> /app/init/fix_permissions\nRUN   /app/init/fix_permissions
# /usr/sbin/entrypoint.sh -> /app/qloud/entrypoint.sh
if [[ -f Dockerfile ]]; then
    sed -r -i "/RUN.*chmod.*entrypoint.sh/d" Dockerfile
    sed -r -i "/CMD.*entrypoint.sh/d" Dockerfile

    (grep -q "entrypoint.sh" Dockerfile) && warning "found unexpected usage of entrypoint.sh"

    echo "RUN chmod 744 /app/init/fix_permissions" >> Dockerfile
    echo "RUN /app/init/fix_permissions" >> Dockerfile
    echo "" >> Dockerfile
    echo "CMD [\"/app/qloud/entrypoint.sh\"]" >> Dockerfile

    (grep -q "ENTRYPOINT" Dockerfile) && warning "found ENTRYPOINT in Dockerfile; only CMD is allowed"
else
    warning "no Dockerfile in project"
fi

# PROGRAM(${oldname}) -> PROGRAM(app)
if ! (grep -qr "PROGRAM($app)" .); then
    warning "can't find binary name in ya.make"
fi
find . -type f | xargs sed -i "s/PROGRAM($app)/PROGRAM(app)/g"

find . -type f | xargs sed -i -r "s/usr\/sbin\/$app/app\/app/g"

# RECURSE(etc) -> RECURSE(config)
sed -r -i "s/(RECURSE\(.*)etc(.*\))/\1config\2/g" ya.make

# === переименование каталогов

mv etc/$app config
if [[ -f etc/ya.make ]]; then
    mv etc/ya.make config
    warning "moved ./etc/ya.make to ./config; check it manually"
fi

if (ls -la etc/$app/* &>/dev/null); then
    warning "directory ./etc/$app not empty"
else
    rmdir etc/$app
fi

if ! (ls -la etc/$app/* &>/dev/null); then
    rmdir etc
fi

if [[ -f config/sandbox.yml && ! -f config/testing.yml ]]; then
    cp config/sandbox.yml config/testing.yml
else
    warning "found both sandbox.yml and testing.yml"
fi

mkdir -p deploy/app

# deploy/usr/share/pop3/monitoring.d -> deploy/app/monitoring
mv deploy/usr/share/$app/monitoring.d deploy/app/monitoring

# deploy/app/doc/<имя>.md
if [[ -f ${arcroot}/mail/sre/doc/${app}.md ]]; then
    mkdir -p deploy/app/doc
    cp ${arcroot}/mail/sre/doc/${app}.md deploy/app/doc
fi

# deploy/app/doc/welcome
if [[ -f deploy/usr/share/$app/entrypoint.d/welcome ]]; then
    mkdir -p deploy/app/doc
    mv deploy/usr/share/$app/entrypoint.d/welcome deploy/app/doc
fi

# ==== push client
# network-tvm-secret: "PUSH_CLIENT_TVM_SECRET" -> network-tvm-secret-file: /app/secrets/tvm_secret
# PUSH_CLIENT_IDENT -> явно указать
# сложить все конфиги в файлы по окружениям:
# ```
# /app/push-client/production.yml
# /app/push-client/corp.yml
# /app/push-client/testing.yml
# ```

find deploy -type f | xargs sed -r -i 's/tvm-secret: .*/tvm-secret-file: \/app\/secrets\/tvm_secret/g'
if (grep -nqr "PUSH_CLIENT_IDENT" deploy); then
    warning "find PUSH_CLIENT_IDENT environment variable usage; specify push client ident explicitly "
fi

mkdir -p deploy/app/push-client
for file in deploy/etc/*push-client/*; do
    if [[ "$file" =~ "yaml-production" ]]; then
        mv $file deploy/app/push-client/production.yml
    elif [[ "$file" =~ "yaml-corp" ]]; then
        mv $file deploy/app/push-client/corp.yml
    elif [[ "$file" =~ "yaml-sandbox" ]]; then
        mv $file deploy/app/push-client/testing.yml
    else
        if [[ -f "$file" ]]; then
            warning "unsupported push client config filename: $file"
        else
            warning "no push client configs found"
        fi
    fi
done

# %QLOUD_APPLICATION% -> $app
find . -type f | xargs sed -i "s/%QLOUD_APPLICATION%/$app/g"

# === final checks

others=$(find deploy -type f | grep -v -e "deploy/app" -e "deploy/etc/cron.d" -e "deploy/etc/cron.yandex" -e "deploy/etc/monrun" -e "deploy/juggler/checks/local")
for file in $others; do
    warning "don't know what to do with $file"
done

warn_patterns=$(cat << EOT
QLOUD_APPLICATION
etc/$app
var/log/$app
usr/share/$app
usr/sbin
EOT
)

for pattern in $warn_patterns; do
    (grep -nr $pattern) && warning "found \"$pattern\"; check it manually"
done

[[ -d deploy/etc/cron.d || -d deploy/etc/cron.yandex ]] && warning "check cron scripts manually"
[[ -d deploy/etc/monrun ]] && warning "found deploy/etc/monrun; check juggler checks manually, perhaps should place to deploy/juggler/checks/local"

for file in $(find deploy/app/monitoring -type f | grep ".py"); do
    head -n1 $file | grep -qw python && \
    warning "found monitoring script with py2: $file; migrate it to py3, keep in mind bytes and strings"
done

popd
