import argparse
import re
import requests
from db_list import get_sharpei_url, get_acquired_db_list, get_hosts


def db_has_live_instances(db):
    try:
        return any([d['status'] == 'alive' for d in db['databases']])
    except:
        return True


def get_sharpei_dbs(sharpei_url):
    url = '{}/v2/stat'.format(sharpei_url)
    db_ids = set()
    r = requests.get(url, timeout=5)
    for name, description in r.json().items():
        if db_has_live_instances(description):
            db_ids.add(name)
    return db_ids


def get_db_list():
    sharpei_url = get_sharpei_url()
    dbs = set() if sharpei_url is None else get_sharpei_dbs(sharpei_url)
    return dbs


def main():
    parser = argparse.ArgumentParser(description="Shards analyzer script")
    parser.add_argument("--host_distrib", action='store_true')
    args = parser.parse_args()

    with_hosts_distribution = args.host_distrib
    try:
        hosts = get_hosts()
        known_dbs = get_db_list()
        acquired_dbs = get_acquired_db_list(hosts)
        acquired_dbs_list = set(acquired_dbs.keys())
        print('{}/{} shards are acquired'.format(len(acquired_dbs_list), len(known_dbs)))

        missing_dbs = known_dbs - acquired_dbs_list
        excrescent_dbs = acquired_dbs_list - known_dbs
        if missing_dbs and excrescent_dbs:
            print(
                '{} not acquired, excrecent {}'.format(
                    ' '.join(missing_dbs),
                    ' '.join(excrescent_dbs))
            )
        elif missing_dbs:
            print('{} not acquired!'.format(' '.join(missing_dbs)))
        elif excrescent_dbs:
            print('{} is excrescent!'.format(' '.join(excrescent_dbs)))
        else:
            print('OK! all dbs acquired')

        if with_hosts_distribution:
            acquired_dbs_by_hosts = {}

            for shard, host in acquired_dbs.items():
                dc = re.search(r"^(?:(?P<dc>[a-z]{3})[0-9]*-)?", host).group("dc") or "UNKNOWN"
                acquired_dbs_by_hosts.setdefault(dc, dict()).setdefault(host, set()).add(shard)

            for dc, hosts in acquired_dbs_by_hosts.items():
                print(dc.upper())
                for host, shards in hosts.items():
                    print('{} acquires {} shards: {}'.format(host, len(shards), ' '.join(shards)))

    except Exception as e:
        print('{}: {}'.format(type(e).__name__, str(e)))


if __name__ == '__main__':
    main()
