#!/usr/bin/env python3

import asyncio
from itertools import chain, zip_longest
from typing import List, Iterable

import click
import ujson
from aiopg.sa import create_engine

import uvloop
from aiohttp.web_response import Response
from aiohttp.web_response import json_response
from mail.python.theatre.app.hostname import gethostname
from mail.python.theatre.profiling.typing import Metric, Metrics
from mail.python.theatre.stages.base import Stage
from mail.python.theatre.stages.bucket_holder.roles.director import Director as BucketHolderDirector
from mail.python.theatre.stages.db_stats.roles.director import Director as DbStatsDirector
from mail.samsara.samsara.interactions.db.meta import shards_metainfo
from mail.samsara.samsara.roles.buckets_provider import BucketsProvider
from .config import load_config_file
from .settings.app import Settings
from .sharpei import get_shard


class DbStatsStage(Stage):
    def __init__(self, env: str, config_path: str, worker_name: str):
        self.config = load_config_file(config_path)
        self.worker_name = worker_name or gethostname()
        super().__init__(env=env, name='samsara', settings=Settings)

    async def shard_dbstats_factory(self, shard_id: int):
        db = get_shard(self.config['sharpei'], shard_id, user=self.config['db_user'])
        return DbStatsDirector(
            [db],
            self.config['signals'],
            self.config['db_host_status_poll_time'],
            fatPollersConfig=None,
        )

    async def create_app(self):
        pg_mngr = create_engine(dsn=self.settings.db.pg_dsn(username='samsara'), maxsize=1)
        pg = await pg_mngr.__aenter__()
        db_meta = shards_metainfo()
        buckets_provider = BucketsProvider(huskydb_pg=pg, meta=db_meta, settings=self.settings.buckets_provider)
        director = BucketHolderDirector(
            pg,
            self.worker_name,
            factory=self.shard_dbstats_factory,
            settings=self.settings,
            db_meta=db_meta,
        )
        self.app.update(settings=self.settings, director=director, buckets_provider=buckets_provider)
        await buckets_provider.start()
        await director.start()

        async def shutdown(_):
            await buckets_provider.stop()
            await director.stop()
            await pg_mngr.__aexit__(None, None, None)

        self.app.on_shutdown.append(shutdown)
        self.setup_routes()
        return self.app

    def setup_routes(self):
        self.app.router.add_get('/ping', lambda _: Response(text='pong\n'))
        self.app.router.add_get('/unistat', self.unistat_handler, name='unistat')

    @staticmethod
    def reorder_stats(stats_iter: Iterable[Metrics]) -> Iterable[Metric]:
        """
        Flat-mapping sequence of metric sequences into single sequence, ordering them on round-robin basis.
        That is, for metric sequences a = [a1, a2, a3], b = [b1, b2, b3], c = [c1], result will be
        reorder_stats([a,b,c]) = [a1, b1, c1, a2, b2, a3, b3]
        That allows to return more valuable metrics from every processed shard first,
        decreasing probability for that metrics to be thrown away by yasm-agent.
        """
        class Sentinel:
            pass

        return (
            sig for sig in chain.from_iterable(
                stats for stats in zip_longest(*list(stats_iter), fillvalue=Sentinel)
            )
            if sig is not Sentinel
        )

    async def unistat_handler(self, _):
        shard_processors: List[DbStatsDirector] = self.app['director'].processor.processors.values()
        return json_response(
            list(self.reorder_stats(shard_proc.stats for shard_proc in shard_processors)),
            dumps=ujson.dumps
        )


@click.command()
@click.option('--host', default='::0', help='IP addr to bind to [= "::0"]')
@click.option('--port', default=8080, help='HTTP port to listen [= 8080]')
@click.option('--env', help='Environment name', required=True)
@click.option('--config-path', help='Path to congig file', required=True)
@click.option('--worker-name', help='Worker name to be used for shard acquisition')
def main(host, port, env, config_path, worker_name):
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())
    stage = DbStatsStage(env, config_path, worker_name)
    stage.run(host=host, port=port)
