package ru.yandex.ace.ventura.proxy;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.UserType;
import ru.yandex.ace.ventura.proxy.config.ImmutableAceVenturaProxyConfig;
import ru.yandex.ace.ventura.proxy.feedback.SuggestReportHandler;
import ru.yandex.ace.ventura.proxy.feedback.SuggestReportQueue;
import ru.yandex.ace.ventura.proxy.fetch.contact.SearchByContactIdHandler;
import ru.yandex.ace.ventura.proxy.fetch.email.SearchByEmailHandler;
import ru.yandex.ace.ventura.proxy.list.ListContactsHandler;
import ru.yandex.ace.ventura.proxy.search.SearchHandler;
import ru.yandex.ace.ventura.proxy.suggest.SuggestHandler;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class AceVenturaProxy
    extends UniversalSearchProxy<ImmutableAceVenturaProxyConfig>
{
    private final Cache<AceVenturaPrefix, SharedList[]> sharedCache;

    private static final long MINUTES_EXPIRE = 10;
    private static final long MAX_SIZE = 500000;

    private final UpstreamStater upstreamStater;
    private final TimeFrameQueue<Long> cacheHit;
    private final AsyncClient mailSearchClient;
    private final AsyncClient producerIndexClient;
    private final SuggestReportQueue suggestReportQueue;

    public AceVenturaProxy(
        final ImmutableAceVenturaProxyConfig config)
        throws IOException
    {
        super(config);

        producerIndexClient = client("ProducerStore", config.suggestReport());

        mailSearchClient = client("MailSearch", config.mailSearch());

        this.sharedCache =
            CacheBuilder.newBuilder()
                .expireAfterWrite(MINUTES_EXPIRE, TimeUnit.MINUTES)
                .maximumSize(MAX_SIZE)
                .concurrencyLevel(config().workers()).build();

        this.cacheHit = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                cacheHit,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "shared-lists-cache-hit_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "shared-lists-cache-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));

        this.upstreamStater =
            new UpstreamStater(
                config().metricsTimeFrame(),
                "shared-requests");
        registerStater(this.upstreamStater);

        if (config.suggestReport().queueSize() > 0) {
            suggestReportQueue = new SuggestReportQueue(this);
        } else {
            suggestReportQueue = null;
        }

        this.register(
            new Pattern<>("/v1/suggest", true),
            new SuggestHandler(this));

        this.register(
            new Pattern<>("/v1/suggestReport", true),
            new SuggestReportHandler(this, suggestReportQueue));

        this.register(
            new Pattern<>("/v1/search", true),
            new SearchHandler(this));
        this.register(
            new Pattern<>("/v1/search_by_email", true),
            new SearchByEmailHandler(this));

        this.register(
            new Pattern<>("/v1/search_by_contact_id", true),
            new SearchByContactIdHandler(this));

        this.register(
            new Pattern<>("/v1/list_contacts", true),
            new ListContactsHandler(this));
    }

    public AceVenturaPrefix parsePrefix(
        final CgiParams params)
        throws BadRequestException
    {
        UserType utype = params.getEnum(UserType.class, "user_type");
        long uid = params.getLong("user_id");
        return new AceVenturaPrefix(uid, utype);
    }

    public Cache<AceVenturaPrefix, SharedList[]> sharedCache() {
        return sharedCache;
    }

    public UpstreamStater upstreamStater() {
        return upstreamStater;
    }

    public TimeFrameQueue<Long> cacheHit() {
        return cacheHit;
    }

    public AsyncClient mailSearchProxyClient() {
        return mailSearchClient;
    }

    public ImmutableHttpHostConfig mailSearchConfig() {
        return config.mailSearch();
    }

    public AsyncClient producerIndexClient() {
        return producerIndexClient;
    }

    @Override
    public void start() throws IOException {
        super.start();

        suggestReportQueue.start();
    }

    @Override
    public void close() throws IOException {
        suggestReportQueue.close();
        super.close();
    }
}
