package ru.yandex.ace.ventura.proxy.common;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.AceVenturaProxyContext;
import ru.yandex.ace.ventura.proxy.SharedList;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.search.request.SearchBackendRequestType;
import ru.yandex.search.rules.pure.SearchRule;

public abstract class AbstractSharedRule<
        M extends AceVenturaProxyContext,
        T extends AceVenturaProxyContextProvider<M>,
        R>
        implements SearchRule<T, R> {
    protected final AceVenturaProxy proxy;
    protected final R empty;

    protected AbstractSharedRule(final AceVenturaProxy proxy, final R empty) {
        this.proxy = proxy;
        this.empty = empty;
    }

    @Override
    public void execute(
            final T input,
            final FutureCallback<? super R> callback)
            throws HttpException
    {
        M context = input.searchContext();
        if (context.shareScope() == SharedScope.EXCLUDE) {
            callback.completed(empty);
            return;
        }
        SharedListFetcher.fetch(
            context,
            new FetchSharedListsCallback(context, callback),
            proxy,
            SearchBackendRequestType.SEQUENTIAL);
    }

    private class FetchSharedListsCallback extends AbstractFilterFutureCallback<SharedList[], R> {
        final M context;

        protected FetchSharedListsCallback(
                final M context,
                final FutureCallback<? super R> callback) {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(SharedList[] lists) {
            Map<AceVenturaPrefix, List<Long>> shareMap =
                    new LinkedHashMap<>();

            for (SharedList list : lists) {
                shareMap.computeIfAbsent(
                        list.prefix(),
                        (x) -> new ArrayList<>()).add(list.listId());
            }

            if (shareMap.size() == 0) {
                context.logger().fine("No shared lists found");
                callback.completed(empty);
                return;
            }

            try {
                execute(context, shareMap, callback);
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    protected abstract void execute(
            final M context,
            final Map<AceVenturaPrefix, List<Long>> listMap,
            FutureCallback<? super R> callback)
            throws HttpException;
}
