package ru.yandex.ace.ventura.proxy.common;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.proxy.suggest.corp.Scorer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.PositionSavingContainerFactory;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.Parser;

public class AceVenturaContact implements JsonValue {
    private static final CollectionParser<Long, Set<Long>, Exception>
        TAGS_IDS_PARSER =
        new CollectionParser<>(
            MapKeyParser.INSTANCE, LinkedHashSet::new, '\n');

    private final AceVenturaPrefix user;
    private final long contactId;
    private final long listId;
    private final String revision;
    private final Set<Long> tagsIds;
    private final JsonMap vcard;
    private final Set<AceVenturaEmail> emails;
    private final Set<String> emailsStringSet;
    private long lastUsage = 0L;
    private int corpDepDist = 20;
    protected volatile float score = -1;
    protected volatile boolean shared = false;

    public AceVenturaContact(
        final AceVenturaPrefix user,
        final long contactId,
        final long listId,
        final String revision,
        final Set<Long> tagsIds,
        final JsonMap vcard)
    {
        this.user = user;
        this.contactId = contactId;
        this.listId = listId;
        this.revision = revision;
        this.tagsIds = tagsIds;
        this.vcard = vcard;
        this.emails = new LinkedHashSet<>();
        this.emailsStringSet = new LinkedHashSet<>();
    }

    public float score(final Scorer scorer) {
        if (score < 0) {
            score = scorer.score(this);
        }

        return score;
    }

    public float score() {
        return score;
    }

    public void calculateRanks() {
        for (AceVenturaEmail email: emails) {
            if (email.lastUsage() > lastUsage) {
                lastUsage = email.lastUsage();
            }

            if (email.corpDestDist() < corpDepDist) {
                corpDepDist = email.corpDestDist();
            }
        }
    }

    public synchronized void addEmail(final AceVenturaEmail email) {
        this.emails.add(email);
    }

    public synchronized void addAllEmails(
        final Collection<AceVenturaEmail> emails)
    {
        for (AceVenturaEmail email: emails) {
            this.addEmail(email);
        }
    }

    public static CollectionParser<Long, Set<Long>, Exception> tagsIdsParser() {
        return TAGS_IDS_PARSER;
    }

    public AceVenturaPrefix user() {
        return user;
    }

    public long contactId() {
        return contactId;
    }

    public long listId() {
        return listId;
    }

    public String revision() {
        return revision;
    }

    public Set<Long> tagsIds() {
        return tagsIds;
    }

    public JsonMap vcard() {
        return vcard;
    }

    public Set<AceVenturaEmail> emails() {
        return emails;
    }

    public Set<String> emailsStringSet() {
        return emailsStringSet;
    }

    public int corpDepDist() {
        return corpDepDist;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof AceVenturaContact)) {
            return false;
        }
        AceVenturaContact that = (AceVenturaContact) o;
        return contactId == that.contactId &&
            user.equals(that.user);
    }

    @Override
    public int hashCode() {
        return Objects.hash(user, contactId);
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        if (emails.size() > 0) {
            JsonList emailsList = new JsonList(PositionSavingContainerFactory.INSTANCE);
            for (AceVenturaEmail email : emails) {
                JsonMap map =
                    new JsonMap(BasicContainerFactory.INSTANCE);
                map.put("email", new JsonString(email.email()));
                emailsList.add(map);
            }
            vcard.put("emails", emailsList);
        }

        writer.startObject();
        writer.key("contact_owner_user_id");
        writer.value(user.uid());
        writer.key("contact_owner_user_type");
        writer.value(user.userType());
        writer.key("contact_id");
        writer.value(contactId);
//        writer.key("corp_dep_dist");
//        writer.value(corpDepDist);
//        writer.key("last_usage");
//        writer.value(lastUsage);
//        writer.key("score");
//        writer.value(score);
        writer.key("list_id");
        writer.value(listId);
        writer.key("revision");
        writer.value(revision);
        if (shared) {
            writer.key("shared");
            writer.value(shared);
        }
        writer.key("tag_ids");
        writer.value(tagsIds);
        writer.key("emails");
        writer.value(emails);
        writer.key("vcard");
        writer.value((JsonValue) vcard);
        writer.endObject();
    }

    @Override
    public String toString() {
        return JsonType.NORMAL.toString(this);
    }

    public static AceVenturaContact fromLuceneResponse(
        final Logger logger,
        final AceVenturaPrefix prefix,
        final JsonMap resp)
        throws JsonException
    {
        return fromLuceneResponse(logger, prefix, false, null, resp);
    }

    public boolean shared() {
        return shared;
    }

    public AceVenturaContact shared(boolean shared) {
        this.shared = shared;
        return this;
    }

    public long lastUsage() {
        return lastUsage;
    }

    public static AceVenturaContact fromLuceneResponse(
        final Logger logger,
        final AceVenturaPrefix prefix,
        final boolean english,
        final JsonMap resp)
        throws JsonException
    {
        return fromLuceneResponse(logger, prefix, english, null, resp);
    }

    public static AceVenturaContact fromLuceneResponse(
        final Logger logger,
        final AceVenturaPrefix prefix,
        final Collection<String> vcardFields,
        final JsonMap resp)
        throws JsonException
    {
        return fromLuceneResponse(logger, prefix, false, vcardFields, resp);
    }

    public static AceVenturaContact fromLuceneResponse(
        final Logger logger,
        final AceVenturaPrefix prefix,
        final boolean english,
        final Collection<String> vcardFields,
        final JsonMap resp)
        throws JsonException
    {
        Long cid = resp.getLong(AceVenturaFields.EMAIL_CID.stored(), null);
        if (cid == null) {
            cid = resp.getLong(AceVenturaFields.CID.stored());
        }

        long listId = resp.getLong(AceVenturaFields.LIST_ID.stored(), -1L);
        String vcardStr =
            resp.getString(AceVenturaFields.VCARD.stored(), "{}");
        JsonMap vcard;
        try {
            vcard = TypesafeValueContentHandler.parse(vcardStr).asMap();
        } catch (JsonException je) {
            logger.log(Level.WARNING, "Bad vcard " + vcardStr, je);
            vcard = new JsonMap(BasicContainerFactory.INSTANCE);
        }

        if (english) {
            String enNames = resp.getString(AceVenturaFields.EN_NAMES.stored(), null);
            if (enNames != null && !enNames.isEmpty()) {
                String[] split = enNames.split("\t");
                vcard = vcard.deepCopy();
                JsonList namesList = new JsonList(BasicContainerFactory.INSTANCE);
                JsonMap nameMap = new JsonMap(BasicContainerFactory.INSTANCE);
                nameMap.put("first", new JsonString(split[0]));
                if (split.length > 1) {
                    nameMap.put("last", new JsonString(split[1]));
                }

                nameMap.put("middle", new JsonString(""));
                namesList.add(nameMap);
                vcard.put("names", namesList);
                logger.warning("english and put enNames" + JsonType.NORMAL.toString(vcard));
            } else {
                logger.warning("english but no enNames " + JsonType.NORMAL.toString(resp));
            }
        }

        if (vcardFields != null) {
            JsonMap filteredVcard = new JsonMap(BasicContainerFactory.INSTANCE);
            for (String field : vcardFields) {
                filteredVcard.put(
                    field,
                    vcard.getOrDefault(field, JsonNull.INSTANCE));
            }
            vcard = filteredVcard;
        }

        String revision =
            resp.getString(AceVenturaFields.REVISION.stored(), "");

        String tagsStr =
            resp.getString(AceVenturaFields.TAGS.stored(), "");
        Set<Long> tags = Collections.emptySet();
        if (!tagsStr.isEmpty()) {
            try {
                tags = TAGS_IDS_PARSER.apply(tagsStr);
            } catch (Exception e) {
                throw new JsonException("Failed to parse tags " + tagsStr, e);
            }
        }

        return new AceVenturaContact(
            prefix,
            cid,
            listId,
            revision,
            tags,
            vcard);
    }

    private enum MapKeyParser implements Parser<Long> {
        INSTANCE;

        @Override
        public Long parse(final String value) throws Exception {
            return Long.parseLong(value.split("\\n")[0]);
        }
    }
}
