package ru.yandex.ace.ventura.proxy.common;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.util.string.StringUtils;

public class AceVenturaEmail implements JsonValue {
    // { id: 1, value: 'email@yandex.ru', tags: [1,3] }
    private final Long id;
    private final Collection<Integer> tags;
    private final String email;
    private final int corpDestDist;
    private final String groupKey;
    private Long lastUsage;

    public AceVenturaEmail(
        final Long id,
        final Collection<Integer> tags,
        final String email,
        final Long lastUsage)
    {
        this.id = id;
        this.tags = tags;
        this.email = email;
        this.lastUsage = lastUsage;
        this.corpDestDist = Integer.MAX_VALUE;
        this.groupKey = email;
    }

    public AceVenturaEmail(
        final Long id,
        final Collection<Integer> tags,
        final String email,
        final Long lastUsage,
        final String dedupKey,
        final int corpDepDist)
    {
        this.id = id;
        this.tags = tags;
        this.email = email;
        this.lastUsage = lastUsage;
        this.corpDestDist = corpDepDist;
        this.groupKey = dedupKey;
    }

    public String groupKey() {
        return groupKey;
    }

    public static AceVenturaEmail fromLuceneResponse(
        final JsonMap emailObj)
        throws JsonException
    {
        String email = emailObj.getString(AceVenturaFields.EMAIL.stored());
        String login = emailObj.getString(AceVenturaFields.LOGIN.stored(), null);
        String domainNt = emailObj.getString(AceVenturaFields.DOMAIN_NT.stored(), null);
        if (login == null || domainNt == null) {
            int sep = MailAliases.emailSeparatorPos(email);
            if (sep <= 0) {
                login = email;
                domainNt = email;
            } else {
                Map.Entry<String, String> entry = MailAliases.INSTANCE.parseAndNormalize(email, sep);
                login = entry.getKey();
                domainNt = entry.getValue();
            }
        }

        String tagsStr =
            emailObj.getString(AceVenturaFields.TAGS.stored(), "");
        Long id = emailObj.getLong(AceVenturaFields.EMAIL_ID.stored());
        Long lastUsage =
            emailObj.getLong(AceVenturaFields.LAST_USAGE.stored(), 0L);
        Integer corpDepDist = emailObj.getInt("dep_score", Integer.MAX_VALUE);
        Set<Integer> tags = Collections.emptySet();
        if (!tagsStr.isEmpty()) {
            try {
                tags = new LinkedHashSet<>();
                String[] split = tagsStr.split("\n");
                for (String tagStr: split) {
                    if (!tagStr.isEmpty()) {
                        tags.add(Integer.parseInt(tagStr));
                    }
                }
            } catch (NumberFormatException nfe) {
                throw new JsonException("Failed to parse tags " + tagsStr, nfe);
            }
        }

        String dedupKey = StringUtils.concat(login, '@', domainNt);
        return new AceVenturaEmail(id, tags, email, lastUsage, dedupKey, corpDepDist);
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writer.key("id");
        writer.value(id);
        writer.key("value");
        writer.value(email);
        writer.key("last_usage");
        writer.value(lastUsage);
        writer.key("tags");
        writer.value(tags);
        writer.endObject();
    }

    public Long id() {
        return id;
    }

    public Collection<Integer> tags() {
        return tags;
    }

    public String email() {
        return email;
    }

    public Long lastUsage() {
        return lastUsage;
    }

    public void updateLastUsage(final Long lastUsage) {
        if (lastUsage != null && lastUsage > this.lastUsage) {
            this.lastUsage = lastUsage;
        }
    }

    public int corpDestDist() {
        return corpDestDist;
    }

    @Override
    public String toString() {
        return "{email='" + email + '\'' +
            ", lastUsage=" + lastUsage +
            ", corpDestDist=" + corpDestDist + '}';
    }
}
