package ru.yandex.ace.ventura.proxy.common;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.AceVenturaRecordType;
import ru.yandex.ace.ventura.UserType;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.AceVenturaProxyContext;
import ru.yandex.ace.ventura.proxy.SharedList;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.SearchBackendRequestType;


public abstract class SharedListFetcher {
    private static final String LUCENE_REQUEST =
            AceVenturaFields.RECORD_TYPE.prefixed() + ':'
                    + AceVenturaRecordType.SHARED.fieldValue();

    private static BasicAsyncRequestProducerGenerator buildRequestGenerator(
            final AceVenturaProxyContext context)
            throws HttpException {
        QueryConstructor query =
                new QueryConstructor(
                        "/search-shared-lists?IO_PRIO=0&json-type=dollar");

        query.append("service", context.user().service());
        query.append("text", LUCENE_REQUEST);
        query.append(
                "get",
                AceVenturaFields.SHARED_OWNER_UTYPE.stored() + ','
                        + AceVenturaFields.SHARED_OWNER_UID.stored() + ','
                        + AceVenturaFields.SHARED_LIST_ID.stored());
        query.append("prefix", context.user().prefix().toStringFast());
        return new BasicAsyncRequestProducerGenerator(query.toString());
    }

    private static SharedList[] fromJson(final JsonObject o) throws JsonException {
        JsonMap map = o.asMap();
        JsonList hits = map.getList("hitsArray");
        SharedList[] lists = new SharedList[hits.size()];
        int i = 0;
        for (JsonObject jo: hits) {
            JsonMap tagObj = jo.asMap();
            long listId =
                    tagObj.getLong(AceVenturaFields.SHARED_LIST_ID.stored());
            long uid =
                    tagObj.getLong(
                            AceVenturaFields.SHARED_OWNER_UID.stored());
            UserType userType =
                    tagObj.getEnum(
                            UserType.class,
                            AceVenturaFields.SHARED_OWNER_UTYPE.stored());
            AceVenturaPrefix owner =
                    new AceVenturaPrefix(uid, userType);

            lists[i++] = new SharedList(owner, listId);
        }
        return lists;
    }

    public static void fetch(
            final AceVenturaProxyContext context,
            final FutureCallback<SharedList[]> callback,
            final AceVenturaProxy proxy,
            final SearchBackendRequestType requestType)
            throws HttpException {
        SharedList[] lists =
                proxy.sharedCache().getIfPresent(context.prefix());

        if (lists != null) {
            context.logger().info("Loaded lists from cache");
            proxy.cacheHit().accept(1L);
            callback.completed(lists);
            return;
        }

        context.logger().info(
            "Shared lists cache miss");

        proxy.cacheHit().accept(0L);


        ListsExtractionCallback lex = new ListsExtractionCallback(callback, proxy, context);
        if (requestType == SearchBackendRequestType.PARALLEL) {
            context.proxy().parallelRequest(
                    context.session(),
                    context,
                    SharedListFetcher.buildRequestGenerator(context),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    lex);
        } else {
            context.proxy().sequentialRequest(
                    context.session(),
                    context,
                    SharedListFetcher.buildRequestGenerator(context),
                    null,
                    false,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    lex);
        }

    }


    private static class ListsExtractionCallback extends AbstractFilterFutureCallback<JsonObject, SharedList[]> {
        private final AceVenturaProxy proxy;
        private final AceVenturaProxyContext context;

        ListsExtractionCallback(
                FutureCallback<SharedList[]> callback,
                AceVenturaProxy proxy,
                AceVenturaProxyContext context) {
            super(callback);
            this.proxy = proxy;
            this.context = context;
        }

        @Override
        public void completed(JsonObject response) {
            try {
                SharedList[] lists = SharedListFetcher.fromJson(response);
                proxy.sharedCache().put(context.prefix(), lists);
                context.logger().info("Loaded lists to cache");

                callback.completed(lists);
            } catch (JsonException e) {
                failed(e);
            }
        }

        @Override
        public void failed(Exception e) {
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }

}
