package ru.yandex.ace.ventura.proxy.feedback;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.charset.UnsupportedCharsetException;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.ParseException;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;

import ru.yandex.ace.ventura.AceVenturaConstants;
import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.SharedList;
import ru.yandex.ace.ventura.proxy.common.SharedListFetcher;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.SearchBackendRequestType;
import ru.yandex.search.request.util.SearchRequestText;

public class SuggestReportHandler implements ProxyRequestHandler {
    private static final String FUNCTION = "function";
    private static final String ARGS = "args";
    private static final String GET = "get";
    private static final int KEEP_REQUESTS_CNT = 10;

    private final SuggestReportQueue queue;
    private final AceVenturaProxy proxy;

    public SuggestReportHandler(
            final AceVenturaProxy proxy,
            final SuggestReportQueue queue) {
        this.proxy = proxy;
        this.queue = queue;
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException {
        FeedbackAceVenturaContext context =
                new FeedbackAceVenturaContext(proxy, session);
        SharedListFetcher.fetch(
                context,
                new SharedListSuggestReportCallback(session),
                proxy,
                SearchBackendRequestType.SEQUENTIAL);
    }

    private static class SuggestReportCallback
            extends AbstractProxySessionCallback<List<Header>> {
        SuggestReportCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final List<Header> response) {
            for (Header acceptHeader: response) {
                if (acceptHeader != null && acceptHeader.getValue() != null) {
                    ContentType contentType;
                    try {
                        contentType = ContentType.parse(acceptHeader.getValue());
                    } catch (ParseException | UnsupportedCharsetException e) {
                        failed(e);
                        return;
                    }

                    if (ContentType.APPLICATION_JSON.getMimeType().equalsIgnoreCase(
                            contentType.getMimeType())) {
                        session.response(
                                HttpStatus.SC_OK,
                                new StringEntity("{}", contentType));
                        return;
                    }
                }
            }
            session.response(HttpStatus.SC_OK);
        }
    }

    private static class HeaderExtractingCallback
            extends AbstractProxySessionCallback<Object> {

        private final FutureCallback<Header> callback;
        protected HeaderExtractingCallback(
                final ProxySession session,
                final FutureCallback<Header> callback
                ) {
            super(session);
            this.callback = callback;
        }

        @Override
        public void completed(Object response) {
            callback.completed(session.request().getFirstHeader(HttpHeaders.ACCEPT));
        }
    }

    private class SharedListSuggestReportCallback extends AbstractProxySessionCallback<SharedList[]> {

        protected SharedListSuggestReportCallback(
                final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(SharedList[] sharedLists) {
            Set<String> prefixes = Arrays.stream(sharedLists)
                            .map(x -> x.prefix().toStringFast())
                            .collect(Collectors.toSet());
            try {
                FeedbackAceVenturaContext context = new FeedbackAceVenturaContext(proxy, session);
                if (prefixes.isEmpty()) {
                    prefixes.add(context.prefix().toStringFast());
                }

                MultiFutureCallback<Header> mfcb = new MultiFutureCallback<>(new SuggestReportCallback(session));

                for (String prefix: prefixes) {
                    BasicAsyncRequestProducerGenerator generator = generateRequest(context, prefix);
                    FutureCallback<Object> callback = new HeaderExtractingCallback(session, mfcb.newCallback());
                    if (queue == null || !queue.offer(session, generator)) {
                        proxy.producerIndexClient().execute(
                                proxy.producerHost(),
                                generator,
                                EmptyAsyncConsumerFactory.ANY_GOOD,
                                context.contextGenerator(),
                                callback);
                    } else {
                        callback.completed(null);
                    }
                }
                mfcb.done();

            } catch (IOException | HttpException e) {
                failed(e);
            }
        }

        private void lastRequests(
                final JsonWriter writer,
                final String request)
                throws IOException {
            writer.key(AceVenturaFields.LAST_REQUESTS.stored());
            writer.startObject();
            writer.key(FUNCTION);
            writer.value("sum_queue");
            writer.key(ARGS);
            writer.startArray();
            writer.value(request);
            writer.startObject();
            writer.key(FUNCTION);
            writer.value(GET);
            writer.key(ARGS);
            writer.startArray();
            writer.value(AceVenturaFields.LAST_REQUESTS.stored());
            writer.endArray();
            writer.endObject();
            writer.value(KEEP_REQUESTS_CNT);
            writer.endArray();
            writer.endObject();
        }

        protected BasicAsyncRequestProducerGenerator generateRequest(
                final FeedbackAceVenturaContext context,
                final String prefix)
                throws IOException, HttpException {
            StringBuilderWriter sbw = new StringBuilderWriter();

            long shard = context.prefix().hash() % SearchMap.SHARDS_COUNT;

            QueryConstructor qc = new QueryConstructor("/update?report");
            qc.append("prefix", prefix);
            qc.append("shard", shard);
            qc.append("service", AceVenturaConstants.ACEVENTURA_QUEUE);
            qc.append("title", context.title());
            qc.append("user-request", context.request());

            if (context.contactId() != null) {
                // update emails
                try (JsonWriter writer = JsonType.DOLLAR.create(sbw)) {
                    writer.startObject();
                    writer.key("query");
                    writer.value(
                            AceVenturaFields.EMAIL.prefixed()
                                    + ':'
                                    + SearchRequestText.fullEscape(context.title(), false));
                    writer.key("prefix");
                    writer.value(prefix);
                    writer.key("docs");
                    writer.startArray();
                    writer.startObject();
                    writer.key(AceVenturaFields.LAST_USAGE.stored());
                    writer.value(context.updateTs());
                    lastRequests(writer, context.request());
                    writer.endObject();
                    writer.endArray();
                    writer.endObject();
                }

                qc.append("contact", context.contactId());

            } else {
                // update tags
                try (JsonWriter writer = JsonType.DOLLAR.create(sbw)) {
                    writer.startObject();
                    writer.key("prefix");
                    writer.value(context.prefix());
                    writer.key("docs");
                    writer.startArray();
                    writer.startObject();
                    writer.key(AceVenturaFields.ID.stored());
                    writer.value(AceVenturaFields.tagUrl(
                            context.tagId().toString(),
                            context.prefix()));
                    writer.key(AceVenturaFields.LAST_USAGE.stored());
                    writer.value(context.updateTs());
                    lastRequests(writer, context.request());
                    writer.endObject();
                    writer.endArray();
                    writer.endObject();
                }

                qc.append("tag", context.tagId());

            }
            return new BasicAsyncRequestProducerGenerator(
                    qc.toString(),
                    sbw.toString(),
                    StandardCharsets.UTF_8);
        }
    }
}
