package ru.yandex.ace.ventura.proxy.feedback;

import java.io.IOException;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.config.SuggestReportConfig;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class SuggestReportQueue implements Runnable, GenericAutoCloseable<IOException>, Stater {
    private static final int SLEEP_DELAY = 200;

    private final Queue<ReportItem> queue;
    private final AceVenturaProxy proxy;
    private final int maxInAir;
    private final Thread thread;
    private final AtomicInteger inAir = new AtomicInteger();
    private final SuggestReportConfig reportConfig;
    private final TimeFrameQueue<Long> tempErrors;
    private final TimeFrameQueue<Long> finalErrors;
    private volatile boolean stop;

    public SuggestReportQueue(final AceVenturaProxy proxy) {
        this.proxy = proxy;

        this.reportConfig = proxy.config().suggestReport();
        this.maxInAir = (int) (1.5 * reportConfig.connections());
        this.queue = new ArrayBlockingQueue<>(proxy.config().suggestReport().queueSize());
        this.thread = new Thread(proxy.getThreadGroup(), this, "SuggestReportQueue");
        this.thread.setDaemon(true);

        tempErrors = new TimeFrameQueue<>(proxy.config().metricsTimeFrame());
        proxy.registerStater(
                new PassiveStaterAdapter<>(
                        tempErrors,
                        new NamedStatsAggregatorFactory<>(
                                "suggest-report-store-temp-errors_ammm",
                                IntegralSumAggregatorFactory.INSTANCE)));

        finalErrors = new TimeFrameQueue<>(proxy.config().metricsTimeFrame());
        proxy.registerStater(
                new PassiveStaterAdapter<>(
                        finalErrors,
                        new NamedStatsAggregatorFactory<>(
                                "suggest-report-store-final-errors_ammm",
                                IntegralSumAggregatorFactory.INSTANCE)));

        proxy.registerStater(this);
    }

    @Override
    public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer) throws E
    {
        statsConsumer.stat("suggest-report-queue-length_axxx", queue.size());
    }

    public void start() {
        thread.start();
    }

    @Override
    public void run() {
        try {
            while (!stop) {
                if (inAir.get() > maxInAir) {
                    Thread.sleep(SLEEP_DELAY);
                    continue;
                }
                ReportItem item = queue.poll();
                if (item == null) {
                    Thread.sleep(SLEEP_DELAY);
                    continue;
                }

                inAir.incrementAndGet();
                proxy.producerIndexClient().execute(
                        reportConfig.host(),
                        item.generator(),
                        EmptyAsyncConsumerFactory.ANY_GOOD,
                        new QueuedReportCallback(item));
            }
        } catch (Exception e) {
            proxy.logger().log(Level.WARNING, "SuggestReportQueue thread stopped", e);
        }
    }

    public boolean offer(final ProxySession session, final BasicAsyncRequestProducerGenerator generator) {
        String sessionId =
                String.valueOf(
                        session.context().getAttribute(HttpServer.SESSION_ID));

        return queue.offer(new ReportItem(generator, sessionId));
    }


    @Override
    public void close() throws IOException {
        stop = true;
    }

    private class QueuedReportCallback implements FutureCallback<Object> {
        private final ReportItem item;

        public QueuedReportCallback(final ReportItem item) {
            this.item = item;
        }

        @Override
        public void completed(final Object o) {
            inAir.decrementAndGet();
            StringBuilder sb = new StringBuilder(100);
            sb.append(item.sessionId());
            sb.append(" suggestReport successfully completed in ");
            sb.append(System.currentTimeMillis() - item.ts());
            proxy.logger().info(sb.toString());
        }

        @Override
        public void failed(final Exception e) {
            inAir.decrementAndGet();
            RequestErrorType errorType = RequestErrorType.ERROR_CLASSIFIER.apply(e);
            if (errorType == RequestErrorType.NON_RETRIABLE || !queue.offer(item)) {
                finalErrors.accept(1L);

                StringBuilder sb = new StringBuilder(100);
                sb.append(item.sessionId());
                sb.append(" suggestReport failed in ");
                sb.append(System.currentTimeMillis() - item.ts());
                sb.append(" et: ");
                sb.append(errorType);
                proxy.logger().info(sb.toString());
            } else {
                tempErrors.accept(1L);
            }
        }

        @Override
        public void cancelled() {
            inAir.decrementAndGet();

            finalErrors.accept(1L);
            StringBuilder sb = new StringBuilder(100);
            sb.append(item.sessionId());
            sb.append(" suggestReport cancelled in ");
            sb.append(System.currentTimeMillis() - item.ts());
            proxy.logger().info(sb.toString());
        }
    }

    private static class ReportItem {
        private final BasicAsyncRequestProducerGenerator generator;
        private final String sessionId;
        private final long ts;

        public ReportItem(
                final BasicAsyncRequestProducerGenerator generator,
                final String sessionId)
        {
            this.generator = generator;
            this.sessionId = sessionId;
            this.ts = System.currentTimeMillis();
        }

        public BasicAsyncRequestProducerGenerator generator() {
            return generator;
        }

        public String sessionId() {
            return sessionId;
        }

        public long ts() {
            return ts;
        }
    }
}
