package ru.yandex.ace.ventura.proxy.fetch.email;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.common.AceVenturaContact;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.search.rules.pure.SearchRule;

public class SearchByEmailHandler implements ProxyRequestHandler {
    private final AceVenturaProxy proxy;

    private final SearchRule<SearchByEmailContext, Map<String, AceVenturaContact>> personalRule;
    private final SearchRule<SearchByEmailContext, Map<String, AceVenturaContact>> sharedRule;

    public SearchByEmailHandler(final AceVenturaProxy proxy) {
        this.proxy = proxy;

        personalRule = new SearchByEmailRule<>(proxy);
        sharedRule = new SharedSearchByEmailRule<>(proxy, personalRule);
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        SearchByEmailContext context = new BasicSearchByEmailContext(proxy, session);
        context.logger().info("Merge Type " + context.mergeType());
        if (context.mergeType() == ContactsMergeType.NONE) {
            DoubleFutureCallback<Map<String, AceVenturaContact>, Map<String, AceVenturaContact>> dfcb
                = new DoubleFutureCallback<>(new ConcatCallback(context, new SearchByEmailPrinter(context)));
            personalRule.execute(context, dfcb.first());
            sharedRule.execute(context, dfcb.second());
        } else {
            personalRule.execute(context, new PersonalCallback(context));
        }
    }

    private static class ConcatCallback extends AbstractFilterFutureCallback<
        Map.Entry<Map<String, AceVenturaContact>, Map<String, AceVenturaContact>>, Collection<Map.Entry<String, AceVenturaContact>>>
    {
        public ConcatCallback(final SearchByEmailContext context, final SearchByEmailPrinter printer) {
            super(printer);
        }

        @Override
        public void completed(
            final Map.Entry<Map<String, AceVenturaContact>, Map<String, AceVenturaContact>> entry)
        {
            List<Map.Entry<String, AceVenturaContact>> result
                = new ArrayList<>(entry.getKey().size() + entry.getValue().size());

            result.addAll(entry.getKey().entrySet());
            result.addAll(entry.getValue().entrySet());

            callback.completed(result);
        }
    }

    private class PersonalCallback
        extends AbstractProxySessionCallback<Map<String, AceVenturaContact>>
    {
        private final SearchByEmailContext context;
        private final SearchByEmailPrinter printer;

        public PersonalCallback(
            final SearchByEmailContext context)
            throws HttpException
        {
            super(context.session());

            this.context = context;
            this.printer = new SearchByEmailPrinter(context);
        }

        @Override
        public void completed(final Map<String, AceVenturaContact> personalResult) {
            if (context.emails().size() == personalResult.size()
                || personalResult.size() >= context.length())
            {
                printer.completed(personalResult.entrySet());
            } else {
                Set<String> left = new LinkedHashSet<>(context.emails());
                left.removeAll(personalResult.keySet());
                try {
                    sharedRule.execute(
                        new BasicSearchByEmailContext(
                            context,
                            context.length() - personalResult.size(),
                            left),
                        new JoinResultsCallback(printer, context, personalResult));
                } catch (HttpException e) {
                    failed(e);
                }
            }
        }
    }

    private static final class JoinResultsCallback
        extends AbstractFilterFutureCallback<Map<String, AceVenturaContact>,  Collection<Map.Entry<String, AceVenturaContact>>>
    {
        private final SearchByEmailContext context;
        private final Map<String, AceVenturaContact> personalResult;

        public JoinResultsCallback(
            final FutureCallback<? super Collection<Map.Entry<String, AceVenturaContact>>> callback,
            final SearchByEmailContext context,
            final Map<String, AceVenturaContact> personalResult)
        {
            super(callback);
            this.context = context;
            this.personalResult = personalResult;
        }

        @Override
        public void completed(final Map<String, AceVenturaContact> sharedResult) {
            if (context.debug()) {
                context.logger().fine(
                    "Shared search completed with " + sharedResult);
            }

            Collection<Map.Entry<String, AceVenturaContact>> result;
            switch (context.mergeType()) {
                case PERSONAL:
                    Map<String, AceVenturaContact> persPrefResult =
                        new LinkedHashMap<>(sharedResult.size() + personalResult.size());

                    persPrefResult.putAll(sharedResult);
                    persPrefResult.putAll(personalResult);

                    result = persPrefResult.entrySet();
                    break;
                case SHARED:
                    Map<String, AceVenturaContact> sharedPrefResult =
                        new LinkedHashMap<>(sharedResult.size() + personalResult.size());
                    sharedPrefResult.putAll(personalResult);
                    sharedPrefResult.putAll(sharedResult);

                    result = sharedPrefResult.entrySet();
                    break;
                default:
                    callback.failed(
                        new ServerException(
                            HttpStatus.SC_INTERNAL_SERVER_ERROR,
                            "Illegal Merge type in this branch " + context.mergeType()));
                    return;
            }

            callback.completed(result);
        }
    }
}
