package ru.yandex.ace.ventura.proxy.fetch.email;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.AceVenturaConstants;
import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.common.AceVenturaContact;
import ru.yandex.ace.ventura.proxy.common.AceVenturaEmail;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.rules.pure.SearchRule;

public class SearchByEmailRule<T extends SearchByEmailContextProvider>
    implements SearchRule<T, Map<String, AceVenturaContact>>
{
    private static final String OR = " OR ";
    private static final String AND = " AND ";

    private static final String LEFT_JOIN_DP =
        "left_join(" + AceVenturaFields.EMAIL_CID.stored()
            + ',' + AceVenturaFields.CID.prefixed()
            // get/out fields
            + ",," + AceVenturaFields.VCARD.stored()
            + ',' + AceVenturaFields.TAGS.stored()
            + ',' + AceVenturaFields.REVISION.stored()
            + ',' + AceVenturaFields.LIST_ID.stored() + ')';

    private final AceVenturaProxy proxy;

    public SearchByEmailRule(AceVenturaProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super Map<String, AceVenturaContact>> callback)
        throws HttpException
    {
        SearchByEmailContext context = input.searchContext();

        StringBuilder text = new StringBuilder();
        text.append(AceVenturaFields.EMAIL.prefixed());
        text.append(":(");
        for (String email: context.emails()) {
            text.append(email);
            text.append(OR);
        }

        text.setLength(text.length() - OR.length());
        text.append(')');

        if (!context.listIdFilter().isEmpty()) {
            text.append(AND);
            text.append(AceVenturaFields.LIST_ID.prefixed());
            text.append(":(");
            for (Long listId: context.listIdFilter()) {
                text.append(listId);
                text.append(" ");
            }

            text.setLength(text.length() - 1);
            text.append(')');
        }

        QueryConstructor qc = new QueryConstructor("/search-ace-by-email?");
        qc.append("prefix", context.prefix().toStringFast());
        qc.append("service", AceVenturaConstants.ACEVENTURA_QUEUE);
        qc.append("dp", LEFT_JOIN_DP);
        qc.append("get", "*");
        qc.append("length", context.length());
        qc.append("text",  text.toString());

        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            context.failOverDelay(),
            false,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            new Callback(callback, context));
    }

    private class Callback
        extends AbstractFilterFutureCallback<
        JsonObject, Map<String, AceVenturaContact>>
    {
        private final SearchByEmailContext context;

        public Callback(
            final FutureCallback<? super Map<String, AceVenturaContact>>
                callback,
            final SearchByEmailContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject response) {
            StringBuilder text =
                new StringBuilder(AceVenturaFields.EMAIL_CID.prefixed());
            text.append(":(");
            Map<String, AceVenturaContact> result;

            QueryConstructor qc = new QueryConstructor(
                "/search-ace-by-email?");

            try {
                JsonMap map = response.asMap();
                //long total = map.getLong("hitsCount");

                JsonList hits = map.getList("hitsArray");

                result = new LinkedHashMap<>(hits.size());

                for (JsonObject hit: hits) {
                    JsonMap record = hit.asMap();
                    AceVenturaContact contact =
                        AceVenturaContact.fromLuceneResponse(
                            context.logger(),
                            context.prefix(),
                            record);
                    String emailStr =
                        record.getString(AceVenturaFields.EMAIL.prefixed());
                    if (result.putIfAbsent(emailStr, contact) == null) {
                        text.append(contact.contactId());
                        text.append(OR);
                    }
                }

                text.setLength(text.length() - OR.length());
                text.append(")");

                qc.append("prefix", context.prefix().toStringFast());
                qc.append("service", AceVenturaConstants.ACEVENTURA_QUEUE);
                qc.append("get", "*");
                qc.append("text",  text.toString());

            } catch (BadRequestException | JsonException e) {
                failed(e);
                return;
            }

            if (result.size() == 0) {
                callback.completed(Collections.emptyMap());
            } else {
                proxy.sequentialRequest(
                    context.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(qc.toString()),
                    context.failOverDelay(),
                    false,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.session().listener().adjustContextGenerator(
                        context.proxy().searchClient()
                            .httpClientContextGenerator()),
                    new FetchEmailCallback(context, result, callback));
            }
        }
    }

    private static final class FetchEmailCallback
        extends AbstractFilterFutureCallback<JsonObject, Map<String,
        AceVenturaContact>>
    {
        private final SearchByEmailContext context;
        private final Map<String, AceVenturaContact> emailMap;
        private final Map<Long, AceVenturaContact> cidContact;

        public FetchEmailCallback(
            final SearchByEmailContext context,
            final Map<String, AceVenturaContact> emailMap,
            final FutureCallback<? super Map<String, AceVenturaContact>> callback)
        {
            super(callback);

            this.context = context;
            this.emailMap = emailMap;

            this.cidContact = new LinkedHashMap<>(emailMap.size());

            for (Map.Entry<String, AceVenturaContact> entry
                : emailMap.entrySet())
            {
                cidContact.put(entry.getValue().contactId(), entry.getValue());
            }
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonMap map = response.asMap();
                //long total = map.getLong("hitsCount");
                JsonList hits = map.getList("hitsArray");

                for (JsonObject hit: hits) {
                    JsonMap record = hit.asMap();
                    AceVenturaEmail email =
                        AceVenturaEmail.fromLuceneResponse(record);
                    Long cid =
                        record.getLong(AceVenturaFields.EMAIL_CID.stored());
                    AceVenturaContact contact = cidContact.get(cid);
                    if (contact == null) {
                        context.logger().warning(
                            "Error, email for missing contact id "
                                + emailMap + " "
                                + JsonType.NORMAL.toString(email));
                        continue;
                    }

                    contact.addEmail(email);
                }

                callback.completed(emailMap);
            } catch (JsonException je) {
                failed(je);
            }
        }
    }
}
