package ru.yandex.ace.ventura.proxy.list;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaRecordType;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.common.AceVenturaContact;
import ru.yandex.ace.ventura.proxy.common.AceVenturaEmail;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.rules.pure.SearchRule;

public class ListContactsFetchRule
    implements SearchRule<ListContactsContext, List<AceVenturaContact>>
{
    protected static final String GET_FIELDS =
        AceVenturaFields.TAGS.stored() + ',' +
            AceVenturaFields.REVISION.stored() + ',' +
            AceVenturaFields.VCARD.stored() + ',' +
            AceVenturaFields.LIST_ID.stored() + ',' +
            AceVenturaFields.CID.stored() + ',' +
            AceVenturaFields.EMAIL_CID.stored() + ',' +
            AceVenturaFields.EMAIL.stored() + ',' +
            AceVenturaFields.EMAIL_ID.stored() + ',' +
            AceVenturaFields.LAST_USAGE.stored();

    protected static final String LEFT_JOIN_DP =
        "left_join(" + AceVenturaFields.EMAIL_CID.stored()
            + ',' + AceVenturaFields.CID.prefixed()
            // get/out fields
            + ",," + AceVenturaFields.VCARD.stored()
            + ',' + AceVenturaFields.TAGS.stored()
            + ',' + AceVenturaFields.REVISION.stored()
            + ',' + AceVenturaFields.LIST_ID.stored() + ')';

    private final AceVenturaProxy proxy;
    protected static final String OR = " OR ";
    protected static final String AND = " AND ";

    public ListContactsFetchRule(final AceVenturaProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void execute(
        final ListContactsContext context,
        final FutureCallback<? super List<AceVenturaContact>> callback) throws HttpException
    {
        long failoverDelay = proxy.config().suggest().failoverDelay();
        boolean localityShuffle = proxy.config().suggest().localityShuffle();

        StringBuilder queryText = new StringBuilder();

        queryText.append(AceVenturaFields.RECORD_TYPE.prefixed());
        queryText.append(':');
        queryText.append(AceVenturaRecordType.EMAIL.fieldValue());

        QueryConstructor query =
            new QueryConstructor(
                "/search-ace-list-cntcts?IO_PRIO=0&json-type=dollar");


        int length = context.offset() + 2 * context.length();
        query.append("service", context.user().service());
        query.append("dp", LEFT_JOIN_DP);
        query.append("get", GET_FIELDS);
        query.append("text", queryText.toString());
        query.append("length", length);
        query.append("prefix", context.user().prefix().toStringFast());
        query.append("lowercase-expanded-terms", "true");
        query.append("replace-ee-expanded-terms", "true");

        if (context.sort() == null) {
            query.append("collector", "passthru(" + length + ')');
        } else {
            query.append("sort", context.sort().get());
        }

        if (failoverDelay > 0) {
            context.proxy().sequentialRequest(
                context.session(),
                context,
                new BasicAsyncRequestProducerGenerator(query.toString()),
                failoverDelay,
                localityShuffle,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.contextGenerator(),
                new Callback(context, callback));
        } else {
            context.proxy().parallelRequest(
                context.session(),
                context,
                new BasicAsyncRequestProducerGenerator(query.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.contextGenerator(),
                new Callback(context, callback));
        }
    }

    private static class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<AceVenturaContact>>
    {
        private final ListContactsContext context;
        private final long fetchTime;

        public Callback(
            final ListContactsContext context,
            final FutureCallback<? super List<AceVenturaContact>> callback)
        {
            super(callback);

            this.context = context;
            this.fetchTime = System.currentTimeMillis();
        }

        @Override
        public void completed(final JsonObject response) {
            context.logger().info(
                "Lucene request finished in "
                    + (System.currentTimeMillis() - fetchTime) + "ms");
            try {
                JsonMap map = response.asMap();

                JsonList hits = map.getList("hitsArray");

                if (context.debug()) {
                    context.logger().info(
                        "EmailSuggest lucene response "
                            + JsonType.HUMAN_READABLE.toString(hits));
                }

                Map<Long, AceVenturaContact> resMap = new LinkedHashMap<>();
                for (JsonObject item: hits) {
                    JsonMap emailRecord = item.asMap();
                    String emailStr =
                        emailRecord.getString(AceVenturaFields.EMAIL.stored());
                    if (emailStr == null) {
                        continue;
                    }

                    Long cid =
                        emailRecord.getLong(AceVenturaFields.EMAIL_CID.stored());
                    AceVenturaEmail email =
                        AceVenturaEmail.fromLuceneResponse(emailRecord);

                    AceVenturaContact contact = resMap.get(cid);
                    if (contact == null) {
                        contact =
                            AceVenturaContact.fromLuceneResponse(
                                context.session().logger(),
                                context.prefix(),
                                item.asMap());
                        resMap.put(cid, contact);
                    }

                    contact.addEmail(email);
                }

                callback.completed(new ArrayList<>(resMap.values()));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }
}
